﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;

namespace Nintendo.NintendoSdkVsExtension.Views
{
    // 実はダイアログではなく Tooltip 等別のコンテナで表示してもよい。
    // その場合は、おそらく DialogType ではなく DataTemplate を与えるのがよい。

    /// <summary>
    /// Content に VM が設定されたらダイアログ表示する Behavior 的なもの。
    /// </summary>
    public class DialogPresenter : Freezable
    {
        private Window _dialog;

        /// <summary>
        /// 本インスタンスの親コントロール。Parent が存在するウィンドウが、表示するダイアログの親ウィンドウとなる。
        /// </summary>
        public UIElement Parent
        {
            get { return (UIElement)GetValue(ParentProperty); }
            set { SetValue(ParentProperty, value); }
        }
        public static readonly DependencyProperty ParentProperty =
            DependencyProperty.Register(
                "Parent", typeof(UIElement), typeof(DialogPresenter),
                new PropertyMetadata());

        /// <summary>
        /// 表示するダイアログ。Window のサブクラスであり、デフォルトコンストラクタを持つこと。
        /// </summary>
        public Type DialogType
        {
            get { return (Type)GetValue(DialogTypeProperty); }
            set { SetValue(DialogTypeProperty, value); }
        }
        public static readonly DependencyProperty DialogTypeProperty =
            DependencyProperty.Register(
                "DialogType", typeof(Type), typeof(DialogPresenter),
                new PropertyMetadata());

        /// <summary>
        /// ダイアログに与える VM
        /// </summary>
        public object Content
        {
            get { return GetValue(ContentProperty); }
            set { SetValue(ContentProperty, value); }
        }
        public static readonly DependencyProperty ContentProperty =
            DependencyProperty.Register(
                "Content", typeof(object), typeof(DialogPresenter),
                new FrameworkPropertyMetadata(null, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault, OnContentChanged));

        protected override Freezable CreateInstanceCore()
        {
            return new DialogPresenter();
        }

        private static void OnContentChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            ((DialogPresenter)d).OnContentChanged(e);
        }

        private void OnContentChanged(DependencyPropertyChangedEventArgs e)
        {
            if (e.OldValue == e.NewValue)
            {
                return;
            }

            if (this._dialog != null)
            {
                CloseDialog();
            }

            if (e.NewValue != null)
            {
                OpenDialog();
            }
        }

        private void CloseDialog()
        {
            this._dialog.Close();
            this._dialog = null;
        }

        private void OpenDialog()
        {
            // デザインモードのときはダイアログを開かない
            if (DesignerProperties.GetIsInDesignMode(this))
            {
                return;
            }
            if (!this.DialogType.IsSubclassOf(typeof(Window)))
            {
                throw new InvalidOperationException("DialogType must be a Window subclass");
            }
            var ctor = DialogType.GetConstructor(new Type[0]);
            if (ctor == null)
            {
                throw new InvalidOperationException("DialogType must have default ctor");
            }

            var parentWindow = ViewUtil.GetAncestorElementByType<Window>(this.Parent);

            var d = (Window)ctor.Invoke(new object[0]);
            d.ShowInTaskbar = false;
            d.DataContext = this.Content;
            d.Owner = parentWindow;
            d.WindowStartupLocation = parentWindow != null ? WindowStartupLocation.CenterOwner : WindowStartupLocation.CenterScreen;
            d.Closed += OnDialogClosed;

            this._dialog = d;
            Dispatcher.BeginInvoke(new Action(() =>
            {
                this._dialog.ShowDialog();
            }));
        }

        private void OnDialogClosed(object sender, EventArgs e)
        {
            // ダイアログが閉じられた場合
            // OK の場合は、子 VM から親 VM に OK コマンドが飛び、親 VM が Content を null にすることにより閉じる
            // キャンセルの場合はそれが行われずにここに来る。Content を null にすることで親 VM に通知する（通常、親 VM がハンドリングする必要はない）
            if (this.Content != null)
            {
                this.Content = null;
            }
        }
    }
}
