﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.Shell.Interop;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Nintendo.NintendoSdkVsExtension.Shell
{
    public class SolutionInfo
    {
        public SolutionInfo(string solutionDirectory, string solutionFile, string userOptsFile)
        {
            this.SolutionDirectory = solutionDirectory;
            this.SolutionFile = solutionFile;
            this.UserOptsFile = userOptsFile;
        }

        /// <summary>.sln ファイルの存在するディレクトリのフルパス</summary>
        public string SolutionDirectory { get; }
        /// <summary>.sln ファイルのフルパス</summary>
        public string SolutionFile { get; }
        /// <summary>.suo ファイルのフルパス</summary>
        public string UserOptsFile { get; }
    }

    public static class VsSolutionUtil
    {
        public static bool SolutionHasVcxproj(IServiceProvider serviceProvider)
        {
            return EnumerateVcxproj(serviceProvider).Any();
        }

        public static IEnumerable<IVsHierarchy> EnumerateVcxproj(IServiceProvider serviceProvider) =>
            EnumerateProjectsOfType(serviceProvider, ShellConstants.VcxprojGuid);

        private static IEnumerable<IVsHierarchy> EnumerateProjectsOfType(IServiceProvider serviceProvider, Guid projectType) =>
            EnumerateProjects(serviceProvider, __VSENUMPROJFLAGS.EPF_ALLINSOLUTION | __VSENUMPROJFLAGS.EPF_MATCHTYPE, projectType);

        private static IEnumerable<IVsHierarchy> EnumerateProjects(IServiceProvider serviceProvider, __VSENUMPROJFLAGS flags, Guid guid = new Guid())
        {
            var solution = QuerySolutionService(serviceProvider);

            IEnumHierarchies eh;
            Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
                solution.GetProjectEnum((uint)flags, guid, out eh));
            return EnumerateHierarchies(eh).ToArray();
        }

        private static IEnumerable<IVsHierarchy> EnumerateHierarchies(IEnumHierarchies eh)
        {
            var hierarchy = new IVsHierarchy[1];
            while (true)
            {
                uint fetched;
                Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
                    eh.Next(1, hierarchy, out fetched));
                if (fetched <= 0)
                {
                    break;
                }
                yield return hierarchy[0];
            }
        }

        public static IVsHierarchy GetProjectOfUniqueName(IServiceProvider serviceProvider, string uniqueName)
        {
            IVsSolution solution = QuerySolutionService(serviceProvider);
            IVsHierarchy h;
            Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
                solution.GetProjectOfUniqueName(uniqueName, out h));
            return h;
        }

        //public static IVsHierarchy GetProjectOfGuid(IServiceProvider serviceProvider, Guid guid)
        //{
        //    IVsSolution solution = QuerySolutionService(serviceProvider);
        //    IVsHierarchy h;
        //    Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
        //        solution.GetProjectOfGuid(guid, out h));
        //    return h;
        //}

        public static bool IsSolutionOpen(IServiceProvider serviceProvider) =>
            GetSolutionProperty<bool>(serviceProvider, __VSPROPID.VSPROPID_IsSolutionOpen);

        public static T GetSolutionProperty<T>(IServiceProvider serviceProvider, __VSPROPID propid)
        {
            var solution = QuerySolutionService(serviceProvider);
            object value;
            Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
                solution.GetProperty((int)propid, out value));
            return (T)value;
        }

        public static SolutionInfo GetSolutionInfo(IServiceProvider serviceProvider)
        {
            if (!IsSolutionOpen(serviceProvider))
            {
                throw new InvalidOperationException("There is no open solution.");
            }

            var solution = QuerySolutionService(serviceProvider);
            string solutionDirectory;
            string solutionFile;
            string userOptsFile;
            Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
                solution.GetSolutionInfo(out solutionDirectory, out solutionFile, out userOptsFile));

            return new SolutionInfo(solutionDirectory, solutionFile, userOptsFile);
        }

        public static void OpenSolution(IServiceProvider serviceProvider, string solutionFile)
        {
            if (IsSolutionOpen(serviceProvider))
            {
                throw new InvalidOperationException("There already is a open solution.");
            }

            var solution = QuerySolutionService(serviceProvider);
            Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
                solution.OpenSolutionFile(0, solutionFile));
        }

        /// <summary>
        /// 同期的に、ソリューション中のプロジェクトがロードされた状態にします
        /// </summary>
        public static void EnsureSolutionIsLoaded(IServiceProvider serviceProvider)
        {
            var solution4 = QuerySolutionService4(serviceProvider);
            solution4.EnsureSolutionIsLoaded((uint)__VSBSLFLAGS.VSBSLFLAGS_LoadAllPendingProjects);
        }

        /// <summary>
        /// ソリューション全体を閉じます。
        /// </summary>
        public static void CloseSolution(IServiceProvider serviceProvider, __VSSLNSAVEOPTIONS saveOptions)
        {
            if (!IsSolutionOpen(serviceProvider))
            {
                throw new InvalidOperationException("There is no open solution.");
            }

            var solution = QuerySolutionService(serviceProvider);
            // ソリューション全体を閉じる (pHier == null, docCookie == 0)
            Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
                solution.CloseSolutionElement((uint)saveOptions, null, 0));
        }

        /// <summary>
        /// ソリューション全体を保存します。
        /// </summary>
        public static void SaveSolution(IServiceProvider serviceProvider, __VSSLNSAVEOPTIONS saveOptions)
        {
            if (!IsSolutionOpen(serviceProvider))
            {
                throw new InvalidOperationException("There is no open solution.");
            }

            var solution = QuerySolutionService(serviceProvider);
            // ソリューション全体を保存する (pHier == null, docCookie == 0)
            Microsoft.VisualStudio.ErrorHandler.ThrowOnFailure(
                solution.SaveSolutionElement((uint)saveOptions, null, 0));
        }

        private static IVsSolution QuerySolutionService(IServiceProvider serviceProvider)
        {
            return serviceProvider.GetService(typeof(SVsSolution)) as IVsSolution;
        }

        private static IVsSolution4 QuerySolutionService4(IServiceProvider serviceProvider)
        {
            return serviceProvider.GetService(typeof(SVsSolution)) as IVsSolution4;
        }
    }
}
