﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Nintendo.NintendoSdkVsExtension.Base;
using System;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

/// <summary>
/// XmlSerializer に与えてシリアライズ・デシリアライズするためのクラス群
/// </summary>
namespace Nintendo.NintendoSdkVsExtension.Serialization.V1
{
    public class SdkProjectSettingsSerializer
    {
        private static readonly Lazy<XmlSerializer> _Serializer = new Lazy<XmlSerializer>(() => new XmlSerializer(typeof(SerializationSdkProjectSettings)));
        private static readonly Lazy<XmlSchema> _Schema = new Lazy<XmlSchema>(() => LoadSchema());

        public void Serialize(Stream s, SerializationSdkProjectSettings value)
        {
            using (var xw = XmlWriter.Create(s, new XmlWriterSettings() { Encoding = Encoding.UTF8, Indent = true }))
            {
                _Serializer.Value.Serialize(xw, value);
            }
        }

        public SerializationSdkProjectSettings Deserialize(Stream s)
        {
            var readerSettings = new XmlReaderSettings() { ValidationType = ValidationType.Schema };
            readerSettings.Schemas.Add(_Schema.Value);
            using (var xr = XmlReader.Create(s, readerSettings))
            {
                return (SerializationSdkProjectSettings)_Serializer.Value.Deserialize(xr);
            }
        }

        private static XmlSchema LoadSchema()
        {
            var assembly = Assembly.GetExecutingAssembly();
            using (var res = assembly.GetManifestResourceStream("Nintendo.NintendoSdkVsExtension.NintendoSdkProjectSettings.V1.xsd"))
            {
                return XmlSchema.Read(res, null);
            }
        }
    }

    [Serializable]
    [XmlRoot("NintendoSdkProjectSettings")]
    public class SerializationSdkProjectSettings
    {
        [XmlElement(Order = 1)]
        public string Version { get; set; }

        [XmlElement(Order = 2)]
        public SerializationProjectSetting ProjectSetting { get; set; }

        public SerializationSdkProjectSettings()
        {
        }

        public SerializationSdkProjectSettings(SdkProjectSettingsStore s)
        {
            this.Version = BaseConstants.SettingVersion;
            this.ProjectSetting = new SerializationProjectSetting(s);
        }

        public SdkProjectSettingsStore ToStore()
        {
            return this.ProjectSetting.ToStore();
        }
    }

    [Serializable]
    public class SerializationProjectSetting
    {
        [XmlArray("ConfigurationSettings", Order = 1)]
        [XmlArrayItem("ConfigurationSetting", IsNullable = false)]
        public SerializationConfigurationSetting[] ConfigurationSettings { get; set; }

        public SerializationProjectSetting()
        {
        }

        public SerializationProjectSetting(SdkProjectSettingsStore s)
        {
            this.ConfigurationSettings = s.Configurations
                .OrderBy(x => x.Key.ToString())
                .Select(x => new SerializationConfigurationSetting(x.Value))
                .ToArray();
        }

        public SdkProjectSettingsStore ToStore()
        {
            return new SdkProjectSettingsStore(
                this.ConfigurationSettings.Select(x => x.ToStore()));
        }
    }

    [Serializable]
    public class SerializationSdkRoot
    {
        [XmlElement(Order = 1)]
        public SdkRootKind Kind { get; set; }

        [XmlElement(Order = 2)]
        public string Value { get; set; }

        public SerializationSdkRoot()
        {
        }

        public SerializationSdkRoot(SdkRoot store)
        {
            this.Kind = store.Kind;
            this.Value = store.Value;
        }

        public SdkRoot ToStore()
        {
            return new SdkRoot(this.Kind, this.Value);
        }
    }

    [Serializable]
    public partial class SerializationConfigurationSetting
    {
        [XmlElement(Order = 1)]
        public string Name { get; set; }

        [XmlElement(Order = 2)]
        public bool IsSdkEnabled { get; set; }

        [XmlElement(Order = 3)]
        public SerializationSdkRoot SdkRoot { get; set; }

        [XmlElement(Order = 4)]
        public string Spec { get; set; }

        [XmlElement(Order = 5)]
        public SdkBuildType BuildType { get; set; }

        public SerializationConfigurationSetting()
        {
        }

        public SerializationConfigurationSetting(SdkConfigurationSettingsStore store)
        {
            this.Name = store.ConfigurationPair.ToString();
            this.IsSdkEnabled = store.IsSdkEnabled;
            this.SdkRoot = new SerializationSdkRoot(store.SdkRoot);
            this.Spec = store.Spec;
            this.BuildType = store.BuildType;
        }

        public SdkConfigurationSettingsStore ToStore()
        {
            return new SdkConfigurationSettingsStore(new ConfigurationPair(this.Name))
            {
                IsSdkEnabled = this.IsSdkEnabled,
                SdkRoot = this.SdkRoot.ToStore(),
                Spec = this.Spec,
                BuildType = this.BuildType,
            };
        }
    }
}
