﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.NintendoSdkVsExtension.Common
{
    public abstract class BindableBase : INotifyPropertyChanged
    {
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// フィールドの値を設定し、値に変化があれば PropertyChanged イベントを発生させる。
        /// </summary>
        protected bool SetProperty<TValue>(ref TValue storage, TValue value, [CallerMemberName] string propertyName = null)
        {
            if (EqualityComparer<TValue>.Default.Equals(storage, value))
            {
                return false;
            }

            storage = value;
            RaisePropretyChanged(propertyName);
            return true;
        }

        /// <summary>
        /// setter で値を設定し、値に変化があれば PropertyChanged イベントを発生させる。典型的にはプロパティに使用する。
        /// </summary>
        protected bool SetProperty<TThis, TValue>(TThis target, Action<TThis, TValue> setter, TValue currentValue, TValue newValue, [CallerMemberName] string propertyName = null)
        {
            if (EqualityComparer<TValue>.Default.Equals(currentValue, newValue))
            {
                return false;
            }

            setter(target, newValue);
            RaisePropretyChanged(propertyName);
            return true;
        }

        protected void RaisePropretyChanged([CallerMemberName]string propertyName = null)
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }
    }
}
