﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.NintendoSdkVsExtension.Base
{
    /// <summary>
    /// １ つのプロジェクトの NintendoSDK 設定です。(nnsdk.xml ファイルに対応)
    /// </summary>
    public class SdkProjectSettingsStore
    {
        private Dictionary<ConfigurationPair, SdkConfigurationSettingsStore> _configurationSettings;
        private IReadOnlyDictionary<ConfigurationPair, SdkConfigurationSettingsStore> _readonlyWrapper;

        public IEnumerable<ConfigurationPair> ConfigurationPairs => _configurationSettings.Keys;
        public IReadOnlyDictionary<ConfigurationPair, SdkConfigurationSettingsStore> Configurations =>
            _readonlyWrapper ?? (_readonlyWrapper = new ReadOnlyDictionary<ConfigurationPair, SdkConfigurationSettingsStore>(_configurationSettings));

        public SdkProjectSettingsStore()
        {
            this._configurationSettings = new Dictionary<ConfigurationPair, SdkConfigurationSettingsStore>();
        }

        public SdkProjectSettingsStore(IEnumerable<SdkConfigurationSettingsStore> configurations)
        {
            this._configurationSettings = configurations.ToDictionary(x => x.ConfigurationPair, x => x);
        }

        /// <summary>
        /// other の deep copy として構築します。
        /// </summary>
        public SdkProjectSettingsStore(SdkProjectSettingsStore other)
        {
            this._configurationSettings = other._configurationSettings.ToDictionary(
                x => x.Key, x => new SdkConfigurationSettingsStore(x.Value));
        }

        public void AddConfiguration(SdkConfigurationSettingsStore settings)
        {
            this._configurationSettings.Add(settings.ConfigurationPair, settings);
        }

        public void RemoveConfiguration(ConfigurationPair configurationPair)
        {
            this._configurationSettings.Remove(configurationPair);
        }
    }

    /// <summary>
    /// １ つの構成の NintendoSDK 設定です。
    /// </summary>
    public class SdkConfigurationSettingsStore
    {
        public ConfigurationPair ConfigurationPair { get; }
        public bool IsSdkEnabled { get; set; } = false;
        public SdkRoot SdkRoot { get; set; } = SdkRoot.Empty;
        public string Spec { get; set; } = string.Empty;
        public SdkBuildType BuildType { get; set; } = SdkBuildType.Develop;

        public SdkConfigurationSettingsStore(ConfigurationPair configurationPair)
        {
            this.ConfigurationPair = configurationPair;
        }

        public SdkConfigurationSettingsStore(SdkConfigurationSettingsStore other)
        {
            this.ConfigurationPair = other.ConfigurationPair;
            this.IsSdkEnabled = other.IsSdkEnabled;
            this.SdkRoot = other.SdkRoot;
            this.Spec = other.Spec;
            this.BuildType = other.BuildType;
        }
    }
}
