﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>  // IOV_MAX
#include <cstdio>
#include <cstdlib>

#include <new>
#include <memory>

#include<nn/nn_Abort.h>

#include "./FileInputStream.h"
#include "./FileOutputStream.h"

//
// FileInputStream.h
//
namespace nn { namespace ngc { namespace detail {

FileInputStream::~FileInputStream() NN_NOEXCEPT
{
    if (m_Fd != NLIB_FD_INVALID)
    {
        errno_t e = nlib_fd_close(m_Fd);
        NN_UNUSED(e);
    }
    if (m_IsInternalBuf && m_Buf)
    {
        nlib_free_size(m_Buf, m_BufSize);
        m_Buf = NULL;
    }
}

errno_t FileInputStream::Init(const FileInputStreamSettings& settings) NN_NOEXCEPT
{
    if (m_Buf)
    {
        return EALREADY;
    }
    if (settings.buf)
    {
        m_Buf = reinterpret_cast<unsigned char*>(settings.buf);
        m_IsInternalBuf = false;
    }
    else if (settings.bufferSize > 0)
    {
        m_Buf = reinterpret_cast<unsigned char*>(nlib_malloc(settings.bufferSize));
        if (!m_Buf)
        {
            return ENOMEM;
        }
        m_IsInternalBuf = true;
    }
    else
    {
        return EINVAL;
    }
    // この後例外が発生してはいけない。
    m_BufSize = settings.bufferSize;
    return 0;
}

errno_t FileInputStream::Open(const char* filename) NN_NOEXCEPT
{
    if (!filename)
    {
        return EINVAL;
    }
    if (filename[0] == '\0')
    {
        return EINVAL;
    }
    errno_t e;
    if (m_Fd != NLIB_FD_INVALID)
    {
        return EEXIST;
    }
    if (!m_Buf)
    {
        e = this->Init();
        if (e != 0) return e;
    }

    NLIB_ASSERT(!!*this);
    e = nlib_fd_open(&m_Fd, filename, NLIB_FD_O_RDONLY);
    return e;
}

errno_t FileInputStream::Open(const wchar_t* filename) NN_NOEXCEPT
{
    if (!filename)
    {
        return EINVAL;
    }
    if (filename[0] == '\0')
    {
        return EINVAL;
    }
    errno_t e;
    if (m_Fd != NLIB_FD_INVALID)
    {
        return EEXIST;
    }
    if (!m_Buf)
    {
        e = this->Init();
        if (e != 0) return e;
    }

    char tmpname[512];
    e = nlib_wide_to_utf8(NULL, tmpname, filename);
    if (e != 0)
    {
        return e;
    }

    NLIB_ASSERT(!!*this);
    e = nlib_fd_open(&m_Fd, tmpname, NLIB_FD_O_RDONLY);
    return e;
}

errno_t FileInputStream::FdOpen(nlib_fd fd) NN_NOEXCEPT
{
    if (fd == NLIB_FD_INVALID)
    {
        return EINVAL;
    }
    errno_t e;
    if (m_Fd != NLIB_FD_INVALID)
    {
        return EEXIST;
    }
    if (!m_Buf)
    {
        e = this->Init();
        if (e != 0)
        {
            return e;
        }
    }
    m_Fd = fd;
    NLIB_ASSERT(!!*this);
    return 0;
}

size_t FileInputStream::FillBuffer_(void* p, size_t nbytes) NN_NOEXCEPT
{
    NLIB_ASSERT(p != NULL);
    size_t n;
    errno_t e = nlib_fd_read(&n, m_Fd, p, nbytes);
    if (e != 0)
    {
        this->SetError(e);
        // 0を返した方がよい。
        // でないとReadやPeekが-1を返さなくなる場合が考えられる。
        return 0;
    }
    return n;
}

bool FileInputStream::Close_() NN_NOEXCEPT
{
    if (m_Fd == NLIB_FD_INVALID)
    {
        this->SetError(EBADF);
        return false;
    }
    errno_t e = nlib_fd_close(m_Fd);
    m_Fd = NLIB_FD_INVALID;
    return e == 0;
}

void* FileInputStream::GetWorkBuffer_(size_t* nbytes) NN_NOEXCEPT
{
    if (!m_Buf)
    {
        errno_t e = this->Init();
        if (e != 0)
        {
            *nbytes = 0;
            return NULL;
        }
    }
    *nbytes = m_BufSize;
    return m_Buf;
}

//
// FileOutputStream.h
//
FileOutputStream::~FileOutputStream() NN_NOEXCEPT
{
    if (m_Fd != NLIB_FD_INVALID)
    {
        // NOTE: デストラクタで閉じると、フラッシュやクローズの失敗で
        // 書き込まれていない場合を検知できないから。
        NLIB_ASSERT(m_Fd == NLIB_FD_INVALID);
        bool result = this->Close();
        NN_UNUSED(result);
    }
    if (m_IsInternalBuf && m_Buf)
    {
        nlib_free_size(m_Buf, m_BufSize);
        m_Buf = NULL;
    }
}

errno_t FileOutputStream::Init(const FileOutputStreamSettings& settings) NN_NOEXCEPT
{
    if (m_Buf)
    {
        return EALREADY;
    }

    if (settings.buf)
    {
        m_Buf = reinterpret_cast<unsigned char*>(settings.buf);
        m_IsInternalBuf = false;
    }
    else if (settings.bufferSize > 0)
    {
        m_Buf = reinterpret_cast<unsigned char*>(nlib_malloc(settings.bufferSize));
        if (!m_Buf)
        {
            return ENOMEM;
        }
        m_IsInternalBuf = true;
    }
    else
    {
        return EINVAL;
    }
    this->ResetBuffer(&m_Buf[0], settings.bufferSize);

    // この後例外が発生してはいけない。
    m_BufSize = settings.bufferSize;
    return 0;
}

errno_t FileOutputStream::Open(const char* filename, int flags, int mode) NN_NOEXCEPT
{
    if (!filename)
    {
        return EINVAL;
    }
    if (filename[0] == '\0')
    {
        return EINVAL;
    }
    errno_t e;
    if (m_Fd != NLIB_FD_INVALID)
    {
        return EEXIST;
    }
    if (!m_Buf)
    {
        e = this->Init();
        if (e != 0)
        {
            return e;
        }
    }
    NLIB_ASSERT(!!*this);
    e = nlib_fd_open(&m_Fd, filename, flags, mode);
    return e;
}

errno_t FileOutputStream::Open(const wchar_t* filename, int flags, int mode) NN_NOEXCEPT
{
    if (!filename)
    {
        return EINVAL;
    }
    if (filename[0] == '\0')
    {
        return EINVAL;
    }
    errno_t e;
    if (m_Fd != NLIB_FD_INVALID)
    {
        return EEXIST;
    }
    if (!m_Buf)
    {
        e = this->Init();
        if (e != 0)
        {
            return e;
        }
    }

    char tmpname[512];
    e = nlib_wide_to_utf8(NULL, tmpname, filename);
    if (e != 0)
    {
        return e;
    }

    NLIB_ASSERT(!!*this);
    e = nlib_fd_open(&m_Fd, tmpname, flags, mode);
    return e;
}

errno_t FileOutputStream::FdOpen(nlib_fd fd) NN_NOEXCEPT
{
    if (fd == NLIB_FD_INVALID)
    {
        return EINVAL;
    }
    errno_t e;
    if (m_Fd != NLIB_FD_INVALID)
    {
        return EEXIST;
    }
    if (!m_Buf)
    {
        e = this->Init();
        if (e != 0)
        {
            return e;
        }
    }
    NLIB_ASSERT(!!*this);
    m_Fd = fd;
    return 0;
}

bool FileOutputStream::PushBuffer_(const void* p, size_t nbytes, bool do_flush) NN_NOEXCEPT
{
    if (m_Fd == NLIB_FD_INVALID)
    {
        this->SetError(EBADF);
        return false;
    }

    NLIB_ASSERT(p != NULL);
    NLIB_ASSERT(nbytes > 0);

    size_t total = 0;
    for (;;)
    {
        size_t n;
        errno_t e = nlib_fd_write(&n, m_Fd, reinterpret_cast<const unsigned char*>(p) + total,
                                  nbytes - total);
        if (e != 0)
        {
            this->SetError(e);
            return false;
        }
        total += n;
        if (total == nbytes)
        {
            break;
        }
    }
    if (do_flush)
    {
        errno_t e = nlib_fd_flush(m_Fd);
        if (e != 0)
        {
            return false;
        }
    }
    return true;
}

bool FileOutputStream::Close_() NN_NOEXCEPT
{
    if (m_Fd == NLIB_FD_INVALID)
    {
        this->SetError(EBADF);
        return false;
    }
    errno_t e = nlib_fd_close(m_Fd);
    m_Fd = NLIB_FD_INVALID;
    return e == 0;
}

NN_STATIC_ASSERT(sizeof(nlib_fd_iovec) == sizeof(FdIoVec));
NN_STATIC_ASSERT(sizeof(nlib_fd_iovec::iov_base) == sizeof(FdIoVec::iovBase));
NN_STATIC_ASSERT(sizeof(nlib_fd_iovec::iov_len) == sizeof(FdIoVec::iovLen));

bool FileOutputStream::WriteGather_(const FdIoVec* iov, int iovcnt) NN_NOEXCEPT
{
    if (!this->Flush())
    {
        return false;
    }

    size_t total = 0;
    for (int i = 0; i < iovcnt; ++i)
    {
        total += iov[i].iovLen;
    }
    nlib_fd fd = m_Fd;
    errno_t e;
    size_t nwritten;

    // 最初の書き出しはiovをそのまま使うことができる。
    // nn::ngc::detail::FdIoVec と nlib_fd_iovec の互換がとれているものと仮定して動く
    NN_ABORT_UNLESS(strcmp(typeid(nlib_fd_iovec::iov_base).name(), typeid(FdIoVec::iovBase).name()) == 0);
    NN_ABORT_UNLESS(strcmp(typeid(nlib_fd_iovec::iov_len).name(), typeid(FdIoVec::iovLen).name()) == 0);
    const nlib_fd_iovec* pIov = reinterpret_cast<nlib_fd_iovec*>(const_cast<FdIoVec*>(iov));
    e = nlib_fd_writev(&nwritten, fd, pIov, iovcnt);
    if (e != 0)
    {
        if (e == ENOMEM)
        {
            // 単にメモリが足りない場合はこうすることで成功するかもしれない。
            // WriteGather()にはatomic性は要求されないので、こうしても問題ない。
            return OutputStream::WriteGather(iov, iovcnt);
        }
        this->SetError(e);
        return false;
    }
    if (nwritten == total)
    {
        return true;
    }

    // iovを書き換えるのでコピーを作成する。
    nlib_fd_iovec* myiov = reinterpret_cast<nlib_fd_iovec*>(
        nlib_malloc(sizeof(nlib_fd_iovec) * iovcnt));
    if (!myiov)
    {
        this->SetError(ENOMEM);
        return false;
    }
    nlib_memcpy(myiov, sizeof(nlib_fd_iovec) * iovcnt, iov, sizeof(nlib_fd_iovec) * iovcnt);

    for (;;)
    {
        // nwrittenの分だけ前から差っ引く
        total = total - nwritten;
        for (int i = 0; i < iovcnt; ++i)
        {
            if (nwritten > myiov[i].iov_len)
            {
                nwritten -= myiov[i].iov_len;
                myiov[i].iov_len = 0;
            }
            else
            {
                myiov[i].iov_base = reinterpret_cast<char*>(myiov[i].iov_base) + nwritten;
                myiov[i].iov_len -= nwritten;
                break;
            }
        }
        e = nlib_fd_writev(&nwritten, fd, myiov, iovcnt);
        if (e != 0)
        {
            this->SetError(e);
            nlib_free(myiov);
            return false;
        }
        if (nwritten == total)
        {
            break;
        }
    }

    nlib_free(myiov);
    return true;
}

}}} // nn::ngc::detail
