﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Serialization;
using MakeUnpublishableApiList.ApiListData;

namespace MakeUnpublishableApiList
{
    using PrivateApiTable = Dictionary<string, Dictionary<string, Tuple<string, List<string>>>>;
    using DebugApiTable = Dictionary<string, string>;

    public class Program
    {
        private static string demangleToolPath;
        public static void Main(string[] args)
        {
            if (args.Count() < 3)
            {
                ShowUsage();
                return;
            }

            var apiListFilePath = args[1];
            demangleToolPath = args[2];

            if (!File.Exists(apiListFilePath))
            {
                throw new ArgumentException("API リストが存在しません。");
            }

            if (!File.Exists(demangleToolPath))
            {
                throw new ArgumentException("デマングルツールが存在しません。");
            }

            var document = new XmlDocument();
            document.Load(apiListFilePath);
            var node = document.SelectSingleNode("//api-list");
            if (node == null)
            {
                throw new ArgumentException("API リストに api-list ノードがありません。");
            }

            ApiListModel model;
            using (var stringReader = new StringReader(node.OuterXml))
            {
                var xmlReader = XmlReader.Create(stringReader);

                var serializer = new System.Xml.Serialization.XmlSerializer(typeof(ApiListModel));
                model = (ApiListModel)serializer.Deserialize(xmlReader);
            }

            if (args[0] == "private")
            {
                var table = ParsePrivateApi(model);
                PrintPrivateApi(table, apiListFilePath.Split('\\').Last());
            }
            else if (args[0] == "debug")
            {
                var table = ParseDebugApi(model);
                PrintDebugApi(table, apiListFilePath.Split('\\').Last());
            }
            else
            {
                ShowUsage();
            }

            return;
        }

        private static void ShowUsage()
        {
            var location = Assembly.GetExecutingAssembly().Location;
            var name = Path.GetFileName(location);
            Console.WriteLine($"Usage:{name} <private|debug> apiListFilePath demangleToolPath");
            Console.WriteLine($"e.g. {name} debug debug_api-list_20171016.xml C:\\ToolChains\\clang-for-nx.x.x.x\\nx\\armv7l\\bin\\armv7l-nintendo-nx-eabihf-c++filt.exe");
        }

        private static string GetDemangleStringImpl(string mangleName, string option = "")
        {
            using (var process = new Process())
            {
                process.StartInfo.FileName = demangleToolPath;
                process.StartInfo.Arguments = string.Format("{0} {1}", option, mangleName);
                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardError = true;
                process.StartInfo.RedirectStandardOutput = true;
                process.Start();

                var errorMessage = process.StandardError.ReadToEnd();
                var standardMessage = process.StandardOutput.ReadLine();
                process.WaitForExit();
                if (errorMessage != string.Empty)
                {
                    throw new ArgumentException("デマングルに失敗しました。");
                }
                if (standardMessage == mangleName)
                {
                    standardMessage = string.Empty;
                }

                return standardMessage;
            }
        }

        private static string GetDemangleString(string mangleName)
        {
            var demangleName = GetDemangleStringImpl(mangleName, string.Empty);
            if (demangleName == string.Empty)
            {
                demangleName = GetDemangleStringImpl(mangleName, "-p");
            }
            if (demangleName == string.Empty)
            {
                demangleName = mangleName;
            }

            return demangleName;
        }

        private static List<string> GetApiNameList(string xpath)
        {
            var apiNames = Regex.Replace(xpath, @".*ApiName\[\.=(.+)\]", @"$1");
            apiNames = Regex.Replace(apiNames, @"\s*or\s*\.=", ",").Replace("'", string.Empty);

            return apiNames.Split(',').Distinct().ToList();
        }

        private static PrivateApiTable ParsePrivateApi(ApiListModel model)
        {
            var UnpublishablePrivateApiTable = new PrivateApiTable();

            foreach (var api in model.ApiModel)
            {
                var version = Regex.Replace(api.Xpath, @".*starts-with\([^0-9]*([0-9]+).*\).*", @"$1");
                var apiNameList = GetApiNameList(api.Xpath);

                if (UnpublishablePrivateApiTable.ContainsKey(version) == false)
                {
                    UnpublishablePrivateApiTable.Add(version, new Dictionary<string, Tuple<string, List<string>>>());
                }

                foreach (var mangleName in apiNameList)
                {
                    var key = new Tuple<string, string>(version, mangleName);
                    if (UnpublishablePrivateApiTable[version].ContainsKey(mangleName) == false)
                    {
                        UnpublishablePrivateApiTable[version].Add(mangleName, new Tuple<string, List<string>>(GetDemangleString(mangleName), new List<string>()));
                    }
                    UnpublishablePrivateApiTable[version][mangleName].Item2.Add(api.Title);
                }
            }

            return UnpublishablePrivateApiTable;
        }

        private static void PrintPrivateApi(PrivateApiTable table, string fileName)
        {
            Console.WriteLine(string.Format("        // by {0}", fileName));
            Console.WriteLine("        public static readonly Dictionary<string, Dictionary<string, Tuple<string, List<string>>>> UnpublishablePrivateApiTable =");
            Console.WriteLine("            new Dictionary<string, Dictionary<string, Tuple<string, List<string>>>>()");
            Console.WriteLine("            {");

            foreach (var version in table)
            {
                Console.WriteLine("                {");
                Console.WriteLine(string.Format("                    \"{0}\", // SDKのメジャーバージョン", version.Key));
                Console.WriteLine("                    new Dictionary<string, Tuple<string, List<string>>> // <マングル名,<デマングル名,マングル名に対応するtitleタグのリスト>>");
                Console.WriteLine("                    {");
                foreach (var name in version.Value)
                {
                    var title = string.Join(", ", name.Value.Item2.Select(entry => "\"" + entry + "\"").ToArray());
                    Console.WriteLine(string.Format("                        {{ \"{0}\", new Tuple<string, List<string>>(\"{1}\", new List<string>() {{ {2} }}) }},", name.Key, name.Value.Item1, title));
                }
                Console.WriteLine("                    }");
                Console.WriteLine("                },");
            }
            Console.WriteLine("            };");
        }

        private static DebugApiTable ParseDebugApi(ApiListModel model)
        {
            var UnpublishableDebugApiTable = new DebugApiTable();

            foreach (var api in model.ApiModel)
            {
                var version = Regex.Replace(api.Xpath, @".*starts-with\([^0-9]*([0-9]+).*\).*", @"$1");
                var apiNameList = GetApiNameList(api.Xpath);

                foreach (var mangleName in apiNameList)
                {
                    UnpublishableDebugApiTable.Add(mangleName, GetDemangleString(mangleName));
                }
            }

            return UnpublishableDebugApiTable;
        }

        private static void PrintDebugApi(DebugApiTable table, string fileName)
        {
            Console.WriteLine(string.Format("        // by {0}", fileName));
            Console.WriteLine("        public static readonly Dictionary<string, string> UnpublishableDebugApiTable =");
            Console.WriteLine("            new Dictionary<string, string>()");
            Console.WriteLine("            {");

            foreach (var api in table)
            {
                Console.WriteLine($"                {{ \"{ api.Key }\", \"{ api.Value }\" }},");
            }
            Console.WriteLine("            };");
        }
    }
}
