﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Microsoft.Build.Evaluation;
using Microsoft.Build.Execution;
using Microsoft.VisualStudio.Setup.Configuration;
using MSBuildWrapper.ServiceMessage;

namespace MSBuildWrapper
{
    public class Program
    {
        /// <summary>
        /// 環境変数TMP
        /// </summary>
        private const string EnviromentVariableTmp = "TMP";
        /// <summary>
        /// MSBuildのツールバージョン
        /// </summary>
        private const string MsbuildToolsetVersion = "14.0";
        /// <summary>
        /// TeamCityのログに出力するプロジェクトファイル名
        /// </summary>
        private static string FileName { get; set; }
        /// <summary>
        /// TeamCityのログに出力するテストパターン
        /// </summary>
        /// <note>
        /// BuildVerificationTest({Configuration}|{Platform})
        /// </note>
        private static string TestName { get; set; }
        /// <summary>
        /// TeamCityのログに出力するかどうか
        /// </summary>
        private static bool IsCIMode { get; set; } = true;
        /// <summary>
        /// MSBuildへの引数
        /// </summary>
        private static string[] MsbuildArgs { get; set; }
        /// <summary>
        /// MSBuild.exeが格納されているパス
        /// </summary>
        private static string MsBuildPath { get; set; }
        /// <summary>
        /// 一時フォルダ名
        /// </summary>
        private static string TempFolderName { get; set; }
        /// <summary>
        /// MSBuildのconfiguation(NXかどうかの判断用)
        /// </summary>
        private static string PlatformArg { get; set; }

        /// <summary>
        /// メイン関数
        /// </summary>
        /// <param name="args">プログラム引数</param>
        public static void Main(string[] args)
        {
            try
            {
                // 引数を解析する。
                ParseArgs(args);
            }
            catch (Exception exception)
            {
                var message = $"Failed because argument is invalid. {exception}";
                OutputTeamCityFailureMessage(message);
                return;
            }
            try
            {
                // MSBuildのパスを取得する。
                GetMsBuildPath();
            }
            catch (Exception exception)
            {
                var message = $"Failed because can't find MSBuild.exe. {exception}";
                OutputTeamCityFailureMessage(message);
                return;
            }

            if (IsCIMode)
            {
                Console.WriteLine(TeamCityMessage.EnterMessage(FileName, TestName));
            }

            var resultCode = ExecuteProgram(
                string.Join(" ", MsbuildArgs.Select(
                        target => string.Format("{0}", target))));
            if (resultCode != 0)
            {
                if (IsCIMode)
                {
                    var message = $"MSBuild.exe returned exit code {resultCode}";
                    Console.WriteLine(TeamCityMessage.ExitMessage(FileName, TestName, message));
                }
            }
            else if (IsCIMode)
            {
                Console.WriteLine(TeamCityMessage.ExitMessage(FileName, TestName));
            }
            Environment.Exit(resultCode);
        }
        /// <summary>
        /// 引数を解析する。
        /// </summary>
        /// <param name="args">実行時引数</param>
        private static void ParseArgs(string[] args)
        {
            if (args.Length < 6)
            {
                throw new ArgumentException(
@"More than 5 arguments are required.
The 1st is the project file name
The 2nd is a configuration
The 3rd is a platform
4th, whether to output TeamCity logs
5th, temp folder name
Example) d:\sdk\...\ex.vcxproj VS2013_Debug Win32 true [MSBuild argument]");
            }

            //  先頭の3つはTeamCityログ用の引数で、
            //  4つ目がTMPフォルダ作成用の引数、
            //  5つ目以降がMSBuildへの引数
            FileName = args[0];
            var configuration = args[1];
            var platform = args[2];
            TestName = $"BuildVerificationTest({configuration}|{platform})";
            IsCIMode = (args[3] == "True");
            TempFolderName = args[4];

            MsbuildArgs = new string[args.Length - 5];
            Array.Copy(args, 5, MsbuildArgs, 0, args.Length - 5);
            PlatformArg = platform;
        }
        /// <summary>
        /// MSBuildのパスを取得する。
        /// </summary>
        private static void GetMsBuildPath()
        {
            ISetupConfiguration2 setupConfiguration = null;

            MsBuildPath = string.Empty;
            try
            {
                setupConfiguration = new SetupConfiguration();
                foreach (var inst in EnumerateInstances(setupConfiguration))
                {
                    if (inst.GetDisplayName() == "Visual Studio Professional 2017")
                    {
                        MsBuildPath = Path.Combine(
                                inst.GetInstallationPath(),
                                @"MSBuild\15.0\Bin\MSBuild.exe");
                    }
                }
            }
            catch (Exception)
            {
            }
            if (MsBuildPath == string.Empty)
            {
                var pc = new ProjectCollection();
                if (MsbuildToolsetVersion != string.Empty)
                {
                    if (!pc.ContainsToolset(MsbuildToolsetVersion))
                    {
                        throw new ArgumentException(
                            string.Format($"toolset '{MsbuildToolsetVersion}' is not found"));
                    }
                    pc.DefaultToolsVersion = MsbuildToolsetVersion;
                }

                var buildParam = new BuildParameters(pc);
                MsBuildPath = buildParam.NodeExeLocation;
            }
        }
        /// <summary>
        /// Windowsにインストール済みのプログラム一覧を取得する。
        /// </summary>
        private static IEnumerable<ISetupInstance> EnumerateInstances(
            ISetupConfiguration2 setupConfiguration)
        {
            var enumerator = setupConfiguration.EnumAllInstances();
            var instances = new ISetupInstance[1];
            int count = 0;

            enumerator.Next(instances.Length, instances, out count);
            while (0 < count)
            {
                yield return instances[0];
                enumerator.Next(instances.Length, instances, out count);
            }
        }

        /// <summary>
        /// Prosess()を介して、指定したプログラムを実行する。
        /// </summary>
        /// <param name="argument">実行時引数</param>
        /// <param name="workDirectory">作業フォルダ</param>
        private static int ExecuteProgram(string argument)
        {
            using (var proc = new Process())
            {
                var returnCode = 0;

                try
                {
                    proc.StartInfo = new ProcessStartInfo()
                    {
                        FileName = MsBuildPath,
                        Arguments = argument,
                        WorkingDirectory = Directory.GetCurrentDirectory(),
                        UseShellExecute = false,
                        CreateNoWindow = true,
                        RedirectStandardOutput = true,
                        RedirectStandardError = true,
                    };
                    if (proc.StartInfo.EnvironmentVariables.ContainsKey(EnviromentVariableTmp) == true)
                    {
                        proc.StartInfo.EnvironmentVariables[EnviromentVariableTmp] = TempFolderName;
                    }
                    else
                    {
                        proc.StartInfo.EnvironmentVariables.Add(EnviromentVariableTmp, TempFolderName);
                    }

                    var handler = new DataReceivedEventHandler(
                        (object obj, DataReceivedEventArgs args) =>
                        {
                            if (args.Data == null)
                            {
                                return;
                            }

                            Console.WriteLine(args.Data);
                        });

                    proc.OutputDataReceived += handler;
                    proc.ErrorDataReceived += handler;
                    proc.Start();
                    proc.BeginOutputReadLine();
                    proc.BeginErrorReadLine();
                    proc.WaitForExit();
                    returnCode = proc.ExitCode;
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.ToString());
                    returnCode = 1;
                }
                return returnCode;
            }
        }
        /// <summary>
        /// 引数の解析処理、またはMSBuildのパス取得時に例外発生した場合、
        /// TeamCityのメッセージを表示する。
        /// </summary>
        /// <param name="failureMessage">TeamCityに表示するメッセージ</param>
        private static void OutputTeamCityFailureMessage(string failureMessage)
        {
            if (IsCIMode)
            {
                Console.WriteLine(TeamCityMessage.EnterMessage(FileName, TestName));
                Console.WriteLine(TeamCityMessage.ExitMessage(FileName, TestName, failureMessage));
            }
            Environment.Exit(1);
        }
    }
}
