﻿using System.Text;

namespace IFFRefGen.Src
{
    // スキーマのツリー文字列構築
    public class XsdTreeStringBuilder
    {
        // 文字列の構築
        public static string Build(XsdElement element)
        {
            XsdTreeStringBuilder instance = new XsdTreeStringBuilder();
            instance.BuildElement(element);
            return instance.ToString();
        }

        // コンストラクタ
        private XsdTreeStringBuilder()
        {
            this.Builder = new StringBuilder();
            this.TabString = "  ";
        }

        // element
        private void BuildElement(XsdElement element)
        {
            AppendTab();
            this.Builder.AppendFormat(
                "[E] {0}({1}{2}) : ",
                element.Name,
                (element.IsInitialized ? "i" : " "),
                (element.IsUsed ? "u" : " "));
            XsdComplexType complexType = element.Type as XsdComplexType;
            if (complexType != null)
            {
                BuildComplexType(complexType);
            }
            else
            {
                BuildSimpleType(element.Type as XsdSimpleType);
                this.Builder.Append('\n');
            }
        }

        // complexType
        private void BuildComplexType(XsdComplexType complexType)
        {
            // 型定義
            this.Builder.AppendFormat(
                "[C] {0}({1}{2})",
                complexType.Name,
                (complexType.IsInitialized ? "i" : " "),
                (complexType.IsUsed ? "u" : " "));

            XsdSimpleType baseType = complexType.BaseType as XsdSimpleType;
            if (baseType != null)
            {
                this.Builder.Append(" : ");
                BuildSimpleType(baseType);
            }

            this.Builder.Append('\n');
            this.TabDepth++;

            // アトリビュート
            foreach (XsdAttribute attribute in complexType.Attributes)
            {
                //continue;
                AppendTab();
                this.Builder.AppendFormat(
                    "[A] {0}({1}{2}) : ",
                    attribute.Name,
                    (attribute.IsInitialized ? "i" : " "),
                    (attribute.IsUsed ? "u" : " "));
                BuildSimpleType(attribute.Type);
                this.Builder.Append('\n');
            }

            // コンテンツ
            XsdContentSequence sequence = complexType.Content as XsdContentSequence;
            if (sequence != null) { BuildContentSequence(sequence); }

            XsdContentChoice choice = complexType.Content as XsdContentChoice;
            if (choice != null) { BuildContentChoice(choice); }

            this.TabDepth--;
        }

        // sequence
        private void BuildContentSequence(XsdContentSequence sequence)
        {
            AppendTab();
            this.Builder.AppendFormat("<sequence type=\"{0}\">\n", sequence.ContainerType.Name);

            foreach (XsdContent content in sequence.Contents)
            {
                XsdContentChoice choice = content as XsdContentChoice;
                if (choice != null)
                {
                    this.TabDepth++;
                    BuildContentChoice(choice);
                    this.TabDepth--;
                }

                XsdContentReference reference = content as XsdContentReference;
                BuildElement(reference.Reference);
            }

            AppendTab();
            this.Builder.AppendFormat("</sequence type=\"{0}\">\n", sequence.ContainerType.Name);
        }

        // choice
        private void BuildContentChoice(XsdContentChoice choice)
        {
            AppendTab();
            this.Builder.AppendFormat("<choice type=\"{0}\">\n", choice.ContainerType.Name);

            foreach (XsdContent content in choice.Contents)
            {
                XsdContentSequence sequence = content as XsdContentSequence;
                if (sequence != null)
                {
                    this.TabDepth++;
                    BuildContentSequence(sequence);
                    this.TabDepth--;
                }

                XsdContentReference reference = content as XsdContentReference;
                BuildElement(reference.Reference);
            }

            AppendTab();
            this.Builder.AppendFormat("</choice type=\"{0}\">\n", choice.ContainerType.Name);
        }

        // simpleType
        private void BuildSimpleType(XsdSimpleType simpleType)
        {
            this.Builder.AppendFormat(
                "[S{0}{1}] {2}({3}{4}) : {5}",
                simpleType.IsList ? "L" : " ",
                simpleType.IsEnumeration ? "E" : " ",
                simpleType.Name,
                (simpleType.IsInitialized ? "i" : " "),
                (simpleType.IsUsed ? "u" : " "),
                simpleType.PrimitiveType.Name);
        }

        // タブの追加
        private void AppendTab()
        {
            for (int i = 0; i < this.TabDepth; i++) { this.Builder.Append(this.TabString); }
        }

        // 文字列化
        public override string ToString()
        {
            return this.Builder.ToString();
        }

        private StringBuilder Builder { get; set; }
        private string TabString { get; set; }
        private int TabDepth { get; set; }
    }
}
