﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.IO;
using System.Diagnostics;

namespace IFFRefGen.Src
{
    // HTML サンプルリスト
    public class HtmlSampleList
    {
        // テンプレートファイルの名前
        private readonly string TemplateFileName = "TmplSampleList.html";
        // 保存パスのフォーマット文字列
        private readonly string SavePathFormat = "html/sample/sample_list.html";

        // サンプルリストノードの XPath
        private readonly string SampleListNodeXPath = "//div[@id='contents_list']";
        // サンプルのフォーマット文字列
        private readonly string SampleFormat = "{0}";
        // リンクのフォーマット文字列
        private readonly string HrefFormat = "{0}.html";

        // コンストラクタ
        public HtmlSampleList()
        {
            this.Template = new XmlDocument();
        }

        // 初期化
        public void Initialize()
        {
            // テンプレートの読み込み
            string loadPath = GApp.Config.HtmlOutputConfig.TemplateElementFolder + this.TemplateFileName;
            this.Template.Load(loadPath);
        }

        // 生成
        public void Generate()
        {
            SmplData[] samples = GSmpl.GetSamples();
            if (GApp.Config.SampleSortTable != null)
            {
                Array.Sort<SmplData>(samples,
                    delegate(SmplData lhs, SmplData rhs)
                    {
                        // 拡張子の次にアルファベットソート
                        string lhsExt = Path.GetExtension(lhs.FileName);
                        string rhsExt = Path.GetExtension(rhs.FileName);

                        int lhsIndex = Array.IndexOf<string>(GApp.Config.SampleSortTable, lhsExt);
                        int rhsIndex = Array.IndexOf<string>(GApp.Config.SampleSortTable, rhsExt);
                        if (lhsIndex == rhsIndex) { return lhs.FileName.CompareTo(rhs.FileName); }
                        return lhsIndex - rhsIndex;
                    });
            }

            XmlNode contentsNode = this.Template.SelectSingleNode(this.SampleListNodeXPath);
            XmlElement rootUl = this.Template.CreateElement("ul");
            contentsNode.AppendChild(rootUl);

            XmlElement extensionUl = null;
            string extension = string.Empty;
            foreach (SmplData sample in samples)
            {
                // 拡張子で分類分け
                string sampleExtension = Path.GetExtension(sample.FileName);
                sampleExtension = sampleExtension.Substring(1, sampleExtension.Length - 1);

                if (sampleExtension != extension)
                {
                    XmlElement li = this.Template.CreateElement("li");
                    li.InnerXml = "<strong>" + sampleExtension + "</strong>";
                    rootUl.AppendChild(li);

                    extensionUl = this.Template.CreateElement("ul");
                    li.AppendChild(extensionUl);

                    extension = sampleExtension;
                }

                // サンプルの生成
                GenerateSample(sample, extensionUl);
            }

            // 保存
            string savePath = System.IO.Path.Combine(GApp.Config.HtmlOutputConfig.OutputFolder, this.SavePathFormat);
            GHtml.Save(this.Template, savePath);
        }

        // サンプルの生成
        private void GenerateSample(SmplData sample, XmlNode parentNode)
        {
            XmlElement li = this.Template.CreateElement("li");
            parentNode.AppendChild(li);

            XmlElement a = this.Template.CreateElement("a");
            a.InnerText = string.Format(this.SampleFormat, sample.FileName);
            {
                XmlAttribute href = this.Template.CreateAttribute("href");
                href.InnerText = string.Format(this.HrefFormat, sample.FileName);
                a.Attributes.Append(href);
                XmlAttribute target = this.Template.CreateAttribute("target");
                target.InnerText = "main";
                a.Attributes.Append(target);
            }
            li.AppendChild(a);
        }

        // テンプレート
        public XmlDocument Template { get; private set; }
    }
}
