﻿using System;
using System.Xml;
using System.IO;

namespace IFFRefGen.Src
{
    // HTML リファレンス目次
    public class HtmlReferenceContent
    {
        // テンプレートファイルの名前
        private readonly string TemplateFileName = "TmplReferenceContents.html";

        // 保存パスのフォーマット文字列
        private readonly string SavePathFormat = "html/reference/{0}_contents.html";
        private readonly string SavePathFormatAi = "html/reference/{0}_contents_ai.html";

        // タイトルの XPath
        private readonly string TitleXPath = "//h1[@id='title']";

        // 付加情報の XPath
        private readonly string AdditionalInformationXPath =
            "//p[@id='additional_information']/a";

        // 目次ノードの XPath
        private readonly string ContentsNodeXPath = "//div[@id='contents_tree']";
        // 要素のフォーマット文字列
        private readonly string ElementFormat = "<{0}>";
        // リンクのフォーマット文字列
        private readonly string HrefFormat = "elements/{0}.html";

        // コンストラクタ
        public HtmlReferenceContent(string title, string rootElementName)
        {
            this.Title = title;
            this.RootElementName = rootElementName;
        }

        // 初期化
        public void Initialize()
        {
            // ルート要素の検索
            this.RootElement = GXsd.FindElement(this.RootElementName);
            if (this.RootElement == null)
            {
                throw new Exception("ルート要素が見つかりません " + this.RootElementName);
            }
        }

        // 生成
        public void Generate()
        {
            Generate(false);
            Generate(true);
        }

        public void Generate(bool showAi)
        {
            // テンプレートの読み込み
            XmlDocument doc = new XmlDocument();
            string loadPath = GApp.Config.HtmlOutputConfig.TemplateElementFolder + this.TemplateFileName;
            doc.Load(loadPath);

            // 付加情報表示の生成
            {
                XmlNode addInfoNode = doc.SelectSingleNode(
                    this.AdditionalInformationXPath);
                XmlAttribute href = doc.CreateAttribute("href");
                addInfoNode.Attributes.Append(href);
                if (showAi)
                {
                    addInfoNode.InnerXml += "付加情報非表示";
                    href.InnerText = this.RootElementName + "_contents.html";
                }
                else
                {
                    addInfoNode.InnerXml += "付加情報表示";
                    href.InnerText = this.RootElementName + "_contents_ai.html";
                }
            }

            // タイトルの生成
            XmlNode titleNode = doc.SelectSingleNode(this.TitleXPath);
            titleNode.InnerText = this.Title;

            // 目次の生成
            XmlNode contentsNode = doc.SelectSingleNode(this.ContentsNodeXPath);
            XmlElement ulRoot = doc.CreateElement("ul");
            contentsNode.AppendChild(ulRoot);

            XmlElement li = doc.CreateElement("li");
            ulRoot.AppendChild(li);

            // TODO: 暫定対処でルート要素名直書き。あとで 3D 以外でも作れるように修正
            {
                string elementName = "nw4f_3dif";
                XmlElement anchor = doc.CreateElement("a");
                anchor.InnerText = string.Format(this.ElementFormat, elementName);
                {
                    XmlAttribute href = doc.CreateAttribute("href");
                    href.InnerText = string.Format(this.HrefFormat, elementName);
                    anchor.Attributes.Append(href);
                    XmlAttribute target = doc.CreateAttribute("target");
                    target.InnerText = "main";
                    anchor.Attributes.Append(target);
                }
                li.AppendChild(anchor);
            }

            XmlElement ul = doc.CreateElement("ul");
            li.AppendChild(ul);

            // <file_info> の生成
            GenerateElement(doc, GXsd.FindElement("file_info"), ul, showAi);

            GenerateElement(doc, this.RootElement, ul, showAi);

            // 保存
            if (showAi)
            {
                string savePath = System.IO.Path.Combine(GApp.Config.HtmlOutputConfig.OutputFolder, string.Format(this.SavePathFormatAi,
                    this.RootElementName));

                GHtml.Save(doc, savePath);
            }
            else
            {
                string savePath = System.IO.Path.Combine(GApp.Config.HtmlOutputConfig.OutputFolder, string.Format(this.SavePathFormat,
                    this.RootElementName));

                GHtml.Save(doc, savePath);
            }
        }

        // 要素の生成
        private void GenerateElement(
            XmlDocument doc, XsdElement element, XmlNode parentNode, bool showAi)
        {
            if (!showAi)
            {
                int index = Array.IndexOf<string>(
                    GApp.Config.AdditionalInformationElements, element.Name);
                if (index != -1) { return; }
            }

            XmlElement li = doc.CreateElement("li");
            parentNode.AppendChild(li);

            XmlElement a = doc.CreateElement("a");
            a.InnerText = string.Format(this.ElementFormat, element.Name);
            {
                XmlAttribute href = doc.CreateAttribute("href");
                href.InnerText = string.Format(this.HrefFormat, element.Name);
                a.Attributes.Append(href);
                XmlAttribute target = doc.CreateAttribute("target");
                target.InnerText = "main";
                a.Attributes.Append(target);
            }
            li.AppendChild(a);

            // complexType の場合は子要素を辿る
            XsdComplexType complexType = element.Type as XsdComplexType;
            if ((complexType != null) && (complexType.Content != null))
            {
                XmlElement ul = doc.CreateElement("ul");
                GenerateContent(doc, complexType.Content, ul, showAi);
                if (ul.HasChildNodes) { li.AppendChild(ul); }
            }
        }

        // complexType の内容生成
        private void GenerateContent(
            XmlDocument doc, XsdContent content, XmlNode parentNode, bool showAi)
        {
            // sequence
            XsdContentSequence sequence = content as XsdContentSequence;
            if (sequence != null)
            {
                foreach (XsdContent subContent in sequence.Contents)
                {
                    GenerateContent(doc, subContent, parentNode, showAi);
                }
                return;
            }

            // choice
            XsdContentChoice choice = content as XsdContentChoice;
            if (choice != null)
            {
                foreach (XsdContent subContent in choice.Contents)
                {
                    GenerateContent(doc, subContent, parentNode, showAi);
                }
                return;
            }

            // reference
            XsdContentReference reference = content as XsdContentReference;
            if (reference != null)
            {
                GenerateElement(doc, reference.Reference, parentNode, showAi);
                return;
            }

            // any
            XsdContentAny any = content as XsdContentAny;
            if (any != null) { return; }

            throw new Exception("未知のコンテンツです " + content.Element.Name);
        }

        // タイトル
        public string Title { get; private set; }
        // ルート要素名
        public string RootElementName { get; private set; }
        // ルート要素
        public XsdElement RootElement { get; private set; }
    }
}
