﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.Diagnostics;

namespace IFFRefGen.Src
{
    // 要素の内容説明
    public class HtmlElementContent
    {
        // 内容 XPath
        private readonly string ContentXPath = "//div[@id='content']";
        // 内容無し
        private readonly string ContentEmpty = "内容はありません。";

        // sequence 説明文
        private readonly string SequenceInfo =
            "以下の要素が順番に出現します。[] 内の数値は要素の出現回数です。";

        // choice 説明文
        private readonly string ChoiceInfo =
            "以下の要素のいずれかが出現します。[] 内の数値は要素の出現回数です。";

        // reference 名前
        private readonly string ReferenceNameFormat = "<{0}>";
        // reference リンク
        private readonly string ReferenceLinkFormat = "{0}.html";
        // 同値 Occurs
        private readonly string EqualOccuresFormat = " [{0}]";
        // 無制限 Occurs
        private readonly string UnboundedOccuresFormat = " [{0}～制限無し]";
        // Occurs
        private readonly string OccuresFormat = " [{0}～{1}]";
        // any
        private readonly string AnyFormat =
            "<*>{0} : ユーザー独自のツール用データを定義する要素です。";

        // コンストラクタ
        public HtmlElementContent(XsdElement element, DocElement document, XmlDocument template)
        {
            this.Element = element;
            this.Document = document;
            this.Template = template;
        }

        // 生成
        public void Generate()
        {
            XmlNode contentNode = this.Template.SelectSingleNode(this.ContentXPath);

            if ((this.Document != null) && (this.Document.Content != null))
            {
                XmlElement p = this.Template.CreateElement("p");
                contentNode.AppendChild(p);
                p.InnerXml = GHtml.LinkElement(this.Document.Content);
            }

            XsdComplexType complexType = this.Element.Type as XsdComplexType;
            if (complexType != null)
            {
                GenerateComplexType(contentNode, complexType);
            }
            else
            {
                XsdSimpleType simpleType = this.Element.Type as XsdSimpleType;
                if (simpleType != null)
                {
                    GenerateSimpleType(contentNode, simpleType);
                }
                else
                {
                    throw new Exception(
                        $"element \"{this.Element.Name}\" が complexType でも simpleType でもありません。{this.Element.Type}");
                }
            }

            // 内容無しの処理
            if (contentNode.InnerText == string.Empty)
            {
                XmlElement p = this.Template.CreateElement("p");
                contentNode.AppendChild(p);
                p.InnerText = this.ContentEmpty;
            }
        }

        #region complexType
        // complexType
        public void GenerateComplexType(XmlNode parentNode, XsdComplexType complexType)
        {
            if (complexType.BaseType != null)
            {
                XsdSimpleType simpleType = complexType.BaseType as XsdSimpleType;
                if (simpleType == null)
                {
                    throw new Exception(
                        "extension 元が simpleType でありません" + complexType.Name);
                }
                GenerateSimpleType(parentNode, simpleType);
            }
            if (complexType.Content != null)
            {
                XsdContentSequence sequence = complexType.Content as XsdContentSequence;
                XsdContentChoice choice = complexType.Content as XsdContentChoice;
                if (sequence != null)
                {
                    GenerateSequenceContent(parentNode, sequence);
                }
                else if (choice != null)
                {
                    GenerateChoiceContent(parentNode, choice);
                }
                else
                {
                    throw new Exception("complexType の content が想定外");
                }
            }
        }

        // sequence
        private void GenerateSequenceContent(XmlNode parentNode, XsdContentSequence sequence)
        {
            XmlElement p = this.Template.CreateElement("p");
            p.InnerText = this.SequenceInfo;
            parentNode.AppendChild(p);

            XmlElement ul = this.Template.CreateElement("ul");
            parentNode.AppendChild(ul);

            foreach (XsdContent content in sequence.Contents)
            {
                XsdContentChoice choice = content as XsdContentChoice;
                if (choice != null)
                {
                    XmlElement li = this.Template.CreateElement("li");
                    ul.AppendChild(li);
                    GenerateChoiceContent(li, choice);
                    continue;
                }

                XsdContentReference reference = content as XsdContentReference;
                if (reference != null)
                {
                    GenerateReferenceContent(ul, reference);
                    continue;
                }

                XsdContentAny any = content as XsdContentAny;
                if (any != null)
                {
                    GenerateAnyContent(ul, any);
                    continue;
                }

                throw new Exception("complexType の sequence 以下の content が想定外");
            }
        }

        // choice
        private void GenerateChoiceContent(XmlNode parentNode, XsdContentChoice choice)
        {
            XmlElement p = this.Template.CreateElement("p");
            p.InnerText = this.ChoiceInfo;
            parentNode.AppendChild(p);

            XmlElement ul = this.Template.CreateElement("ul");
            parentNode.AppendChild(ul);

            foreach (XsdContent content in choice.Contents)
            {
                XsdContentSequence sequence = content as XsdContentSequence;
                if (sequence != null)
                {
                    XmlElement li = this.Template.CreateElement("li");
                    ul.AppendChild(li);
                    GenerateSequenceContent(li, sequence);
                    continue;
                }

                XsdContentReference reference = content as XsdContentReference;
                if (reference != null)
                {
                    GenerateReferenceContent(ul, reference);
                    continue;
                }

                XsdContentAny any = content as XsdContentAny;
                if (any != null)
                {
                    GenerateAnyContent(ul, any);
                    continue;
                }

                throw new Exception("complexType の choice 以下の content が想定外");
            }
        }

        // reference
        private void GenerateReferenceContent(XmlNode parentNode, XsdContentReference reference)
        {
            XmlElement li = this.Template.CreateElement("li");
            parentNode.AppendChild(li);

            XmlElement a = this.Template.CreateElement("a");
            li.AppendChild(a);
            XmlAttribute href = this.Template.CreateAttribute("href");
            href.InnerText = string.Format(this.ReferenceLinkFormat, reference.ReferenceName);
            a.Attributes.Append(href);
            a.InnerText = string.Format(ReferenceNameFormat, reference.ReferenceName);

            StringBuilder builder = new StringBuilder();
            if (reference.MinOccurs == reference.MaxOccurs)
            {
                builder.AppendFormat(this.EqualOccuresFormat, reference.MinOccurs);
            }
            else
            {
                if (reference.MaxOccurs == -1)
                {
                    builder.AppendFormat(
                        this.UnboundedOccuresFormat, reference.MinOccurs);
                }
                else
                {
                    builder.AppendFormat(
                        this.OccuresFormat, reference.MinOccurs, reference.MaxOccurs);
                }
            }

            li.AppendChild(this.Template.CreateTextNode(builder.ToString()));

            // 概要説明
            if (GDoc.Elements.ContainsKey(reference.ReferenceName))
            {
                DocElement docElement = GDoc.Elements[reference.ReferenceName];
                if (docElement.Summary != null)
                {
                    li.InnerXml += string.Format(" : {0}",
                        GHtml.LinkElement(docElement.Summary));
                }
            }
        }

        // any
        private void GenerateAnyContent(XmlNode parentNode, XsdContentAny any)
        {
            XmlElement li = this.Template.CreateElement("li");
            parentNode.AppendChild(li);

            StringBuilder builder = new StringBuilder();
            if (any.MinOccurs == any.MaxOccurs)
            {
                builder.AppendFormat(this.EqualOccuresFormat, any.MinOccurs);
            }
            else
            {
                if (any.MaxOccurs == -1)
                {
                    builder.AppendFormat(
                        this.UnboundedOccuresFormat, any.MinOccurs);
                }
                else
                {
                    builder.AppendFormat(
                        this.OccuresFormat, any.MinOccurs, any.MaxOccurs);
                }
            }

            li.InnerText = string.Format(this.AnyFormat, builder.ToString());
        }
        #endregion

        #region simpleType
        // simpleType
        public void GenerateSimpleType(XmlNode parentNode, XsdSimpleType simpleType)
        {
            // 基本的にドキュメントで上書きする
            if ((this.Document != null) && (this.Document.Content != null)) { return; }

            StringBuilder builder = new StringBuilder();
            if (simpleType.IsEnumeration)
            {
                builder.Append("enum ");
            }

            builder.Append(simpleType.PrimitiveType.Name);

            if (simpleType.IsList)
            {
                builder.AppendFormat("[{0}]", simpleType.ListLength);
            }

            if ((simpleType.MinInclusive != string.Empty) ||
                (simpleType.MinExclusive != string.Empty) ||
                (simpleType.MaxInclusive != string.Empty) ||
                (simpleType.MaxExclusive != string.Empty))
            {
                if (simpleType.MinExclusive != string.Empty)
                {
                    builder.AppendFormat("({0}<", simpleType.MinExclusive);
                }
                else
                {
                    builder.AppendFormat("({0}", simpleType.MinInclusive);
                }
                builder.Append("～");
                if (simpleType.MaxExclusive != string.Empty)
                {
                    builder.AppendFormat("<{0})", simpleType.MaxExclusive);
                }
                else
                {
                    builder.AppendFormat("{0})", simpleType.MaxInclusive);
                }
            }

            if (simpleType.HasPattern)
            {
                builder.AppendFormat(" Pattern[{0}]", simpleType.Pattern);
            }

            if (simpleType.HasWhiteSpace)
            {
                builder.AppendFormat(" WhiteSpace[{0}]", simpleType.WhiteSpace);
            }

            if (simpleType.PrimitiveType != simpleType)
            {
                builder.AppendFormat(" {0}", simpleType.Name);
            }

            parentNode.InnerText = builder.ToString();
        }
        #endregion

        // 要素
        public XsdElement Element { get; private set; }
        // 要素ドキュメント
        public DocElement Document { get; private set; }
        // テンプレート
        public XmlDocument Template { get; private set; }
    }
}
