﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace GllSourceGenerator
{
    public interface IInterfaceElements
    {
        List<InterfaceElementType> Type { get; set; }

        List<InterfaceElementEnum> Enum { get; set; }

        List<InterfaceElementCommand> Command { get; set; }
    }

    public interface IFeature
    {
        List<Require> Require { get; set; }

        List<Remove> Remove { get; set; }

        string name { get; set; }
    }

    public partial class Type
    {
        public string Value { get; set; }

        public Type Requires { get; set; }
    }

    public partial class Enums
    {
        public List<Enum> Enum { get; set; }

        public List<Unused> Unused { get; set; }
    }

    public partial class Proto
    {
        public string Value { get; set; }

        public Type Type { get; set; }
    }

    public partial class Param
    {
        public string Value { get; set; }

        public Type Type { get; set; }
    }

    public partial class Feature : IFeature
    {
        public List<Require> Require { get; set; }

        public List<Remove> Remove { get; set; }
    }

    public partial class Extension : IFeature
    {
        public List<Require> Require { get; set; }

        public List<Remove> Remove { get; set; }
    }

    public partial class Require : IInterfaceElements
    {
        public List<InterfaceElementType> Type { get; set; }

        public List<InterfaceElementEnum> Enum { get; set; }

        public List<InterfaceElementCommand> Command { get; set; }
    }

    public partial class Remove : IInterfaceElements
    {
        public List<InterfaceElementType> Type { get; set; }

        public List<InterfaceElementEnum> Enum { get; set; }

        public List<InterfaceElementCommand> Command { get; set; }
    }

    public partial class InterfaceElementType
    {
        public Type Type { get; set; }
    }

    public partial class InterfaceElementEnum
    {
        public Enum Enum { get; set; }
    }

    public partial class InterfaceElementCommand
    {
        public Command Command { get; set; }
    }

    public partial class Registry
    {
        public List<Commands> Commands { get; set; }

        public List<Enums> Enums { get; set; }

        public List<Extensions> Extensions { get; set; }

        public List<Feature> Feature { get; set; }

        public List<Groups> Groups { get; set; }

        public List<Types> Types { get; set; }

        public List<Type> SortedTypes { get; set; }

        public List<Enum> SortedEnums { get; set; }

        public List<Command> SortedCommands { get; set; }

        public void Resolve( string api )
        {
            ResolveObjectArray( this );
            foreach( var types in Types )
            {
                foreach( var type in types.type )
                {
                    var text = ResolveText( false, type.Text, type.apientry, type.name );
                    var lines = text.Split( '\n' );
                    type.Value = string.Empty;
                    for( int idxLine = 0; idxLine < lines.Length; ++idxLine )
                    {
                        var line = lines[ idxLine ];
                        type.Value += line;
                        if( line.StartsWith( "#include" ) )
                        {
                            type.Value += " // NOLINT";
                        }
                        if( idxLine != lines.Length - 1 )
                        {
                            type.Value += "\r\n";
                        }
                    }
                    if( string.IsNullOrEmpty( type.name ) )
                    {
                        type.name = type.name1;
                    }
                    if( type.api == null || type.api == api )
                    {
                        SortedTypes.Add( type );
                    }
                }
            }
            foreach( var commands in Commands )
            {
                foreach( var command in commands.command )
                {
                    command.proto.Value = ResolveText( true,
                        command.proto.Text, command.proto.ptype, command.proto.name );
                    if( command.param != null )
                    {
                        foreach( var param in command.param )
                        {
                            param.Value = ResolveText( true, param.Text, param.ptype, param.name );
                        }
                    }
                    SortedCommands.Add( command );
                }
            }
            foreach( var enums in Enums )
            {
                foreach( var @enum in enums.Enum )
                {
                    SortedEnums.Add( @enum );
                }
            }
            SortedCommands.Sort( ( lhs, rhs ) => string.Compare( lhs.proto.name, rhs.proto.name ) );
            SortedTypes.Sort( ( lhs, rhs ) => string.Compare( lhs.name, rhs.name ) );
            SortedEnums.Sort( ( lhs, rhs ) => string.Compare( lhs.name, rhs.name ) );

            foreach( var type in SortedTypes )
            {
                type.Requires = SearchType( type.requires );
            }
            foreach( var command in SortedCommands )
            {
                command.proto.Type = SearchType( command.proto.ptype );
                if( command.param != null )
                {
                    foreach( var param in command.param )
                    {
                        param.Type = SearchType( param.ptype );
                    }
                }
            }
            Action<IInterfaceElements> ResolveInterfaceElements = ( interfaceElements ) =>
            {
                foreach( var type in interfaceElements.Type )
                {
                    type.Type = SearchType( type.name );
                }
                foreach( var @enum in interfaceElements.Enum )
                {
                    @enum.Enum = SearchEnum( @enum.name );
                }
                foreach( var command in interfaceElements.Command )
                {
                    command.Command = SearchCommand( command.name );
                }
            };
            foreach( var feature in Feature )
            {
                foreach( var require in feature.Require )
                {
                    ResolveInterfaceElements( require );
                }
                foreach( var remove in feature.Remove )
                {
                    ResolveInterfaceElements( remove );
                }
            }
            foreach( var extensions in Extensions )
            {
                if( extensions.extension != null )
                {
                    foreach( var extension in extensions.extension )
                    {
                        foreach( var require in extension.Require )
                        {
                            ResolveInterfaceElements( require );
                        }
                        foreach( var remove in extension.Remove )
                        {
                            ResolveInterfaceElements( remove );
                        }
                    }
                }
            }
        }

        public Type SearchType( string name )
        {
            if( string.IsNullOrEmpty( name ) )
            {
                return null;
            }
            var found = SortedTypes.BinarySearch( new Type { name = name },
                new LambdaComparer<Type>( ( lhs, rhs ) => string.Compare( lhs.name, rhs.name ) ) );
            return found < 0 ? null : SortedTypes[ found ];
        }

        public Enum SearchEnum( string name )
        {
            if( string.IsNullOrEmpty( name ) )
            {
                return null;
            }
            var found = SortedEnums.BinarySearch( new Enum { name = name },
                new LambdaComparer<Enum>( ( lhs, rhs ) => string.Compare( lhs.name, rhs.name ) ) );
            return found < 0 ? null : SortedEnums[ found ];
        }

        public Command SearchCommand( string name )
        {
            if( string.IsNullOrEmpty( name ) )
            {
                return null;
            }
            var dummy = new Command();
            dummy.proto = new Proto();
            dummy.proto.name = name;
            var found = SortedCommands.BinarySearch( dummy,
                new LambdaComparer<Command>( ( lhs, rhs ) => string.Compare( lhs.proto.name, rhs.proto.name ) ) );
            return found < 0 ? null : SortedCommands[ found ];
        }

        private string ResolveText( bool isTextLeft, string[] texts, string element0, string element1 )
        {
            string space = ( element0 != null && element1 != null ) ? " " : string.Empty;
            string element01 = ( element0 ?? string.Empty ) + space + ( element1 ?? string.Empty );
            if( texts == null )
            {
                return element01;
            }
            string ret = string.Empty;
            // 完璧に復元する方法がない
            switch( texts.Length )
            {
                case 1:
                    if( texts[ 0 ].StartsWith( "[" ) ) // glPathGlyphIndexRangeNV 用の特別処理
                    {
                        ret = element01 + texts[ 0 ];
                    }
                    else if( element0 != null && element1 != null )
                    {
                        ret = element0 + texts[ 0 ] + element1;
                    }
                    else
                    {
                        if( isTextLeft )
                        {
                            ret = texts[ 0 ] + element01;
                        }
                        else
                        {
                            ret = element01 + texts[ 0 ];
                        }
                    }
                    break;
                case 2:
                    if( element0 == null )
                    {
                        ret = texts[ 0 ] + element1 + texts[ 1 ];
                    }
                    else
                    {
                        if( element1 == null )
                        {
                            ret = texts[ 0 ] + element0 + texts[ 1 ];
                        }
                        else
                        {
                            ret = isTextLeft ? texts[ 0 ] + element0 + texts[ 1 ] + element1
                                : element0 + texts[ 0 ] + element1 + texts[ 1 ];
                        }
                    }
                    break;
                case 3:
                    ret = texts[ 0 ] + element0 + texts[ 1 ] + element1 + texts[ 2 ];
                    break;
                default:
                    throw new Exception();
            }
            return ret;
        }

        private void ResolveObjectArray( object target )
        {
            if( target == null || target.GetType().Namespace != GetType().Namespace )
            {
                return;
            }
            foreach( var propertyInfo in target.GetType().GetProperties() )
            {
                if( propertyInfo.PropertyType.BaseType == typeof( Array ) )
                {
                    var items = propertyInfo.GetValue( target ) as object[];
                    if( items != null )
                    {
                        foreach( var item in items )
                        {
                            ResolveObjectArray( item );
                        }
                        if( propertyInfo.PropertyType.GetElementType() == typeof( object ) )
                        {
                            foreach( var listProperty in target.GetType().GetProperties().Where(
                                p => p.PropertyType.IsGenericType &&
                                p.PropertyType.GetGenericTypeDefinition() == typeof( List<> ) ) )
                            {
                                var genericType = listProperty.PropertyType.GetGenericArguments()[ 0 ];
                                dynamic list = listProperty.GetValue( target );
                                foreach( var item in items.Where( p => p.GetType() == genericType ) )
                                {
                                    list.Add( (dynamic)item );
                                }
                            }
                        }
                    }
                }
                else
                {
                    ResolveObjectArray( propertyInfo.GetValue( target ) );
                }
            }
        }
    }

    public class LambdaComparer<T> : IComparer<T>
    {
        public LambdaComparer( Func<T, T, int> func )
        {
            this.func = func;
        }
        public int Compare( T lhs, T rhs )
        {
            return func( lhs, rhs );
        }

        private readonly Func<T, T, int> func;
    }
}
