﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Text.RegularExpressions;

using LOG = Common.Logger;
using FILEIO = Common.FileIOWrapper;

namespace HtmlToVsHelp
{
    public partial class VsHelpConvert
    {
        static TocPartBranchTree TocTopPart;
        static TocPartBranchTree[] TocChildPart;
        static Func<bool>[] TocChildPartSortHandler;

        static readonly List<string> topPartFileNames = new List<string>{ "index", "pages", "namespaces", "annotated", "files", "modules" };
        static readonly List<string>[] childPartFileNames = { new List<string>{"^_page_.+", "_result_finder", "deprecated", "retval", "bug", "overloadlist", "^overloadc_.*", "^nvn.*", "^gpu_.+"},
                                                       new List<string>{"^namespace"},
                                                       new List<string>{"classes", "^class", "^singleton*", "^struct", "^union", "hierarchy", "^functions*"},
                                                       new List<string>{"^globals*", "^dir_*", "_(?:8h|8hpp|8cpp)$", "_(?:8h|8hpp|8cpp)_[^_]+$", "_(?:8h|8hpp|8cpp)-example", "examples"},
                                                       new List<string>{"^group_*"},
                                                       new List<string>{".*"} };
        static readonly string[] parentFileNamseOfChildPart = { "pages", "namespaces", "annotated", "files", "modules", "pages"};

        static bool SetTocInformation()
        {
            TocTopPart = new TocPartBranchTree();
            TocChildPart = new TocPartBranchTree[childPartFileNames.Length];
            for(int index = 0; index < TocChildPart.Length; index++)
            {
                TocChildPart[index] = new TocPartBranchTree();
            }
            TocChildPartSortHandler = new Func<bool>[childPartFileNames.Length];
            TocChildPartSortHandler[0] = new Func<bool>(TocChildPart[0].SortPagesPart);
            TocChildPartSortHandler[1] = new Func<bool>(TocChildPart[1].SortNamespacePart);
            TocChildPartSortHandler[2] = new Func<bool>(TocChildPart[2].SortAnnotatedPart);
            TocChildPartSortHandler[3] = new Func<bool>(TocChildPart[3].SortFilesPart);
            TocChildPartSortHandler[4] = new Func<bool>(TocChildPart[4].SortPartDefault);
            TocChildPartSortHandler[5] = new Func<bool>(TocChildPart[5].SortOtherPart);

            List<string> htmlFileNameList = new List<string>();
            htmlFileNameList.AddRange(Common.UtilFs.GetDoxygenHtmlFiles());

            if (TocTopPart.RegisterSelectedFiles(topPartFileNames, ref htmlFileNameList) == false)
            {
                return false;
            }
            if (TocTopPart.SortFiles(TocTopPart.SortPartDefault) == false)
            {
                return false;
            }

            for (int index = 0; index < childPartFileNames.Length; index++)
            {
                List<string> childPartRule = childPartFileNames[index];
                if (TocChildPart[index].RegisterSelectedFiles(childPartRule, ref htmlFileNameList) == false)
                {
                    return false;
                }

                if (TocChildPart[index].SortFiles(TocChildPartSortHandler[index]) == false)
                {
                    return false;
                }

                if (TocTopPart.ConnectToOtherPart(parentFileNamseOfChildPart[index], TocChildPart[index]) == false)
                {
                    return false;
                }
            }

            return true;
        }

        static bool ConvertSocInfoToTable()
        {
            SetVsHelpPageParent("index.html", "-1");

            foreach (TocNode node in TocTopPart)
            {
                List<Common.MultiBranchTreeNode> childList = null;
                node.GetDirectChildList(ref childList);
                foreach (TocNode childNode in childList)
                {
                    SetVsHelpPageParentCore(childNode.Name() + ".html", node.Name() + ".html");
                }
            }
            return true;
        }
    }

    class TocNode : Common.MultiBranchTreeNode
    {
        string fileName;

        public TocNode(string name) : base()
        {
            fileName = name;
        }

        public string Name()
        {
            return fileName;
        }

        public void Dump()
        {
            // LOG.LogLine(LOG.Level.LOG_INFO, "  {0}", fileName);
            System.Diagnostics.Trace.WriteLine(string.Format("  {0}", fileName));
        }
    }

    class TocPartBranchTree : Common.MultiBranchTreeNodeHead
    {
        List<string> registerdFileNameList;
        List<TocNode> registerdFileNodeList;
        private int allFilesCount;

        public TocPartBranchTree()
        {
            registerdFileNameList = new List<string>();
            registerdFileNodeList = new List<TocNode>();
            allFilesCount = 0;
        }

        public int FileNum()
        {
            return allFilesCount;
        }

        public bool RegisterSelectedFiles(List<string> selectRuleList, ref List<string> fileList)
        {
            foreach (string rule in selectRuleList)
            {
                Regex regex = new Regex(rule);
                for (int index = fileList.Count - 1; index >= 0; index--)
                {
                    string filename = Path.GetFileNameWithoutExtension(fileList[index]);
                    if ((IsRegexString(rule) && regex.IsMatch(filename)) || (!IsRegexString(rule) && filename == rule))
                    {
                        // LOG.LogLine("register {0}", filename);
                        registerdFileNameList.Add(filename);
                        fileList.RemoveAt(index);
                        allFilesCount++;
                    }
                }
            }
            registerdFileNameList.Reverse();

            return true;
        }

        public bool SortFiles(Func<bool> sortHandler)
        {
            return sortHandler();
        }

        public bool SortPartDefault()
        {
            // index があるならそれを親として登録しそれ以外はすべて子供にする
            if (registerdFileNameList.Contains("index"))
            {
                TocNode indexNode = new TocNode("index");
                this.AddChild(indexNode);
                registerdFileNameList.Remove("index");

                foreach (string file in registerdFileNameList)
                {
                    TocNode newNode = new TocNode(file);
                    indexNode.AddChild(newNode);
                }
            }
            else
            {
                foreach (string file in registerdFileNameList)
                {
                    TocNode newNode = new TocNode(file);
                    this.AddChild(newNode);
                }
            }
            return true;
        }

        public bool SortPagesPart()
        {
            if (CreateTocNodes() == false)
            {
                return false;
            }

            if (ForcePairGroup1OfRegex("^(_page_[^_]+).+") == false)
            {
                return false;
            }
            if (ForcePairRegex("_page_graphics_for_n_x", "^nvn_.+") == false)
            {
                return false;
            }
            ForceDirectChild("_page_.+");

            ForceDirectChild("_result_finder");
            ForceDirectChild("deprecated");
            ForceDirectChild("retval");
            ForceDirectChild("bug");

            if (ForcePairRegex("overloadlist", "overloadc_.*") == false)
            {
                return false;
            }
            ForceDirectChild("overloadlist");

            ForceDirectChild("^gpu_.*");

            if (SortUnPairedFiles("pages part unpaired file") == false)
            {
                return false;
            }

            return true;
        }

        public bool SortNamespacePart()
        {
            if (CreateTocNodes() == false)
            {
                return false;
            }

            if (ForcePairGroup1OfRegex("^(namespacenn_[^_]+_[^_]+).+") == false)
            {
                return false;
            }

            if (ForcePairRegex("namespacenn", "namespacenn_.*") == false)
            {
                return false;
            }

            if (ForcePairGroup1OfRegex("^(namespacenw_[^_]+_[^_]+).+") == false)
            {
                return false;
            }

            if (ForcePairRegex("namespacenw", "namespacenw_.*") == false)
            {
                return false;
            }

            if (ForcePairGroup1OfRegex("^(namespacenvn_[^_]+_[^_]+).+") == false)
            {
                return false;
            }

            if (ForcePairRegex("namespacenv", "namespacenv_.*") == false)
            {
                return false;
            }
            if (ForcePairRegex("namespacenvn", "namespacenvn_.*") == false)
            {
                return false;
            }

            if (ForcePairRegex("namespacemembers", "namespacemembers_.*") == false)
            {
                return false;
            }

            ForceDirectChild("namespacenn");
            ForceDirectChild("namespacenw");
            ForceDirectChild("namespacenv");
            ForceDirectChild("namespacenvn");
            ForceDirectChild("namespacemembers");

            if (SortUnPairedFiles("namespace part unpaired file") == false)
            {
                return false;
            }

            return true;
        }

        public bool SortAnnotatedPart()
        {
            if (CreateTocNodes() == false)
            {
                return false;
            }

            if (ForcePairGroup1OfRegex("^(classnn_[^_]+_[^_]+)_.+") == false)
            {
                return false;
            }
            if (ForcePairGroup1OfRegex("^(classnvn_[^_]+_[^_]+)_.+") == false)
            {
                return false;
            }
            else if (ForcePairGroup1OfRegex("^(classnvn_[^_]+_[^_]+_[^_]+)_.+") == false)
            {
                return false;
            }
            if (ForcePairGroup2OfRegexAndParentName("^(struct)([^_]+_[^_]+_[^_]+)_.+", "class") == false)
            {
                return false;
            }
            if (ForcePairGroup2OfRegexAndParentName("^(union)([^_]+_[^_]+_[^_]+)_.+", "class") == false)
            {
                return false;
            }
            if (ForcePairRegex("classes", "^classnn_.+") == false)
            {
                return false;
            }
            if (ForcePairRegex("classes", "^classnvn_.+") == false)
            {
                return false;
            }
            if (ForcePairRegex("classes", "^struct.+") == false)
            {
                return false;
            }
            if (ForcePairRegex("classes", "^union.+") == false)
            {
                return false;
            }
            ForceDirectChild("classes");

            ForceDirectChild("hierarchy");

            if (ForcePairRegex("functions", "^functions_.+") == false)
            {
                return false;
            }
            ForceDirectChild("functions");


            if (SortUnPairedFiles("annotated part unpaired file") == false)
            {
                return false;
            }

            return true;
        }

        public bool SortFilesPart()
        {
            if (CreateTocNodes() == false)
            {
                return false;
            }

            if (ForcePairGroup1OfRegex("^(globals).+") == false)
            {
                return false;
            }
            ForceDirectChild("globals");

            ForceDirectChild("^dir.+");

            if (ForcePairRegex("examples", "^(.+)-example$") == false)
            {
                return false;
            }

            ForceDirectChild("^.+_(?:8h|8hpp|8cpp)$");
            ForceDirectChild("^.+_(?:8h|8hpp|8cpp)_[^_]+$");

            ForceDirectChild("examples");

            if (SortUnPairedFiles("files part unpaired file") == false)
            {
                return false;
            }

            return true;
        }

        public bool SortOtherPart()
        {
            foreach (string file in registerdFileNameList)
            {
                TocNode newNode = new TocNode(file);
                PrintCheckLogLine("Other file : {0}", newNode.Name());
                this.AddChild(newNode);
            }
            return true;
        }

        private bool CreateTocNodes()
        {
            foreach (string file in registerdFileNameList)
            {
                TocNode newNode = new TocNode(file);
                registerdFileNodeList.Add(newNode);
            }
            return true;
        }

        private bool ForcePairGroup1OfRegex(string childRegexString)
        {
            Regex childRegex = new Regex(childRegexString);
            foreach (TocNode node in registerdFileNodeList)
            {
                if (node.GetParent() != null)
                {
                    continue;
                }
                Match match = childRegex.Match(node.Name());
                if (match.Success && match.Groups.Count == 2)
                {
                    foreach (TocNode node2 in registerdFileNodeList)
                    {
                        if (node2.Name() == match.Groups[1].Value)
                        {
                            node2.AddChild(node);
                            break;
                        }
                    }
                }

            }
            return true;
        }

        private bool ForcePairGroup2OfRegexAndParentName(string childRegexString, string parentGroup1String)
        {
            Regex childRegex = new Regex(childRegexString);
            foreach (TocNode node in registerdFileNodeList)
            {
                if (node.GetParent() != null)
                {
                    continue;
                }
                Match match = childRegex.Match(node.Name());
                if (match.Success && match.Groups.Count == 3)
                {
                    foreach (TocNode node2 in registerdFileNodeList)
                    {
                        if (node2.Name() == parentGroup1String + match.Groups[2].Value)
                        {
                            node2.AddChild(node);
                            break;
                        }
                    }
                }

            }
            return true;
        }

        private bool ForcePairRegex(string parentString, string childRegexString)
        {
            if (IsRegexString(parentString))
            {
                return false;
            }

            TocNode parentNode = null;
            foreach (TocNode node in registerdFileNodeList)
            {
                if (node.Name() == parentString)
                {
                    parentNode = node;
                    break;
                }
            }
            if (parentNode == null)
            {
                // ignore paring
                return true;
            }
            Regex childRegex = new Regex(childRegexString);
            foreach (TocNode node in registerdFileNodeList)
            {
                if (node.GetParent() != null)
                {
                    continue;
                }
                if (childRegex.IsMatch(node.Name()))
                {
                    parentNode.AddChild(node);
                }
            }
            return true;
        }

        private bool ForcePair(string parentName, string childName)
        {
            TocNode parentNode = null, childNode = null;
            bool[] isSet = {false, false};
            foreach (TocNode node in registerdFileNodeList)
            {
                if (node.Name() == parentName)
                {
                    parentNode = node;
                    isSet[0] = true;
                }
                else if (node.Name() == childName)
                {
                    childNode = node;
                    isSet[1] = true;
                }
                if (isSet[0] == true && isSet[0] == true)
                {
                    break;
                }
            }
            if (isSet[0] == false)
            {
                // ignore pairing
                return true;
            }
            parentNode.AddChild(childNode);
            return true;
        }

        private bool ForcePair(TocNode parentNode, TocNode childNode)
        {
            parentNode.AddChild(childNode);
            return true;
        }

        private bool ForceDirectChild(string childName)
        {
            if (IsRegexString(childName))
            {
                Regex childRegex = new Regex(childName);
                foreach (TocNode node in registerdFileNodeList)
                {
                    if (node.GetParent() != null)
                    {
                        continue;
                    }
                    else if (childRegex.IsMatch(node.Name()))
                    {
                        this.AddChild(node);
                    }
                }
                return true;
            }
            else
            {
                foreach (TocNode node in registerdFileNodeList)
                {
                    if (node.GetParent() != null)
                    {
                        continue;
                    }
                    else if (node.Name() == childName)
                    {
                        this.AddChild(node);
                        return true;
                    }
                }
            }
            return false;
        }

        private bool SortUnPairedFiles(string messageBase = "UnPaired file :")
        {
            foreach (TocNode node in registerdFileNodeList)
            {
                if (node.GetParent() == null)
                {
                    PrintCheckLogLine(messageBase + " {0}", node.Name());
                    this.AddChild(node);
                }
            }
            return true;
        }

        public bool ConnectToOtherPart(string connectFileName, TocPartBranchTree childPart)
        {
            if (childPart.FileNum() <= 0)
            {
                return true;
            }

            foreach (TocNode childNode in this)
            {
                if (childNode.Name() == connectFileName)
                {
                    foreach( TocNode childPartChildNode in childPart)
                    {
                        childNode.AddChild(childPartChildNode);
                    }
                    return true;
                }
            }

            // connectFileName が存在しない場合は index の下につなぐ
            foreach (TocNode childNode in this)
            {
                if (childNode.Name() == "index")
                {
                    foreach (TocNode childPartChildNode in childPart)
                    {
                        childNode.AddChild(childPartChildNode);
                    }
                    return true;
                }
            }

            // index もない場合はエラー
            // MEMO: TocTopPart 以外の TocPartBranchTree の下に接続することは想定していない

            return false;
        }

        private bool IsRegexString(string str)
        {
            return str.Contains("[") || str.Contains("(") || str.Contains("*") || str.Contains("+") || str.Contains("^");
        }

        private void PrintCheckLog(string format, params object[] args)
        {
            if (CmdlineOption.CheckParentPage)
            {
                LOG.Log(LOG.Level.LOG_WARN, format, args);
            }
        }

        private void PrintCheckLogLine(string format, params object[] args)
        {
            if (CmdlineOption.CheckParentPage)
            {
                LOG.LogLine(LOG.Level.LOG_WARN, format, args);
            }
        }

        public void DumpRegisterdFiles()
        {
            // LOG.LogLine(LOG.Level.LOG_INFO, "TocPartBranchTree files :");
            System.Diagnostics.Trace.WriteLine(string.Format("TocPartBranchTree registered files :"));
            foreach (string fileName in registerdFileNameList)
            {
                // LOG.LogLine(LOG.Level.LOG_INFO, "  {0}", fileName);
                System.Diagnostics.Trace.WriteLine(string.Format("  {0}", fileName));
            }
        }

        public void Dump()
        {
            // LOG.LogLine(LOG.Level.LOG_INFO, "TocPartBranchTree files :");
            System.Diagnostics.Trace.WriteLine("TocPartBranchTree files :");
            foreach (TocNode node in this)
            {
                node.Dump();
            }
        }
    }
}
