﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;
using System.Xml;
using System.Text.RegularExpressions;

using LOG = Common.Logger;
using DOXYGEN_XML = Common.ParseDoxygenXml;
using FILEIO = Common.FileIOWrapper;

namespace AddMeta
{
    public class ParseAndWrite
    {
        private struct FuncInfo
        {
            public string RefId;
            public string FunctionName;
            public string NamespaceName;
        }

        private static List<FuncInfo> stringList = new List<FuncInfo>();
        private static List<FuncInfo> overloadList = new List<FuncInfo>();

        private static string doxygenHtmlPath = null;
        private static string doxygenXmlPath = null;

        private static bool CheckFunctionInfo(FuncInfo info)
        {
            foreach (FuncInfo inf in stringList)
            {
                if (inf.NamespaceName == info.NamespaceName && inf.FunctionName == info.FunctionName)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// ドキュメントパスを指定します
        /// </summary>
        /// <param name="doxygenpath">ドキュメントパス(%SIGLO_ROOT%\Documents\Outputs\Api 相当)</param>
        /// <returns>パスが存在して設定できたら true</returns>
        public static bool SetPath(string doxygenpath)
        {
            if (!Directory.Exists(doxygenpath))
            {
                LOG.Log(LOG.Level.LOG_ERROR, "document base path is not found ({0})\n", doxygenpath);
                return false;
            }
            Common.UtilFs.SetVsHelpBaseDirName(CmdlineOption.InputDirectoryName);
            Common.UtilFs.SetVsHelpDirName(CmdlineOption.OutputDirectoryName);
            doxygenHtmlPath = doxygenpath + Common.UtilFs.GetHtmlDirName();
            doxygenXmlPath = doxygenpath + Common.UtilFs.GetXmlDirName();

            // doxygen HTMLファイル存在チェック用ハッシュの初期化
            Common.UtilFs.InitalizeDoxygenHtmlFileNameHash(doxygenHtmlPath);

            return true;
        }

        /// <summary>
        /// doxynge生成ファイルを解析してmetaを追加するファイル名等を取得します
        /// </summary>
        public static void Parse()
        {
            if (doxygenXmlPath == null)
            {
                return;
            }

            if (CmdlineOption.FlagNamespacePage == false)
            {
                // オーバーロードAPIを調査する
                ParseOverloadFunctions();
            }

            // index.xmlを開く
            string indexxmlpath = doxygenXmlPath + "\\" + Common.UtilFs.DoxygenIndexXmlName;
            if (!File.Exists(indexxmlpath))
            {
                LOG.Log(LOG.Level.LOG_ERROR, "index.xml not found!\n");
                return;
            }

            string fileData;
            FILEIO.GetData(out fileData, indexxmlpath);
            using (StringReader sr = new StringReader(fileData))
            using (XmlTextReader reader = new XmlTextReader(sr))
            {
                // doxygenが生成したXMLの解析
                DOXYGEN_XML.XmlReadLoop(reader,
                                            NamespaceHandler, FunctionHandler, StructHandler, DefineHandler,
                                            VariableHandler, EnumHandler, EnumValueHandler, FileHandler);
            }

            if (CmdlineOption.FlagNamespacePage == false)
            {
                // オーバーロードAPIに含まれているAPIを除外して代わりにオーバーロードページを追加する
                AdjustOverloadApiList();
            }
        }

        private static bool NamespaceHandler(DOXYGEN_XML.CompoundMode cmode, string refid, string namespaceName, string functionName)
        {
            if (CmdlineOption.FlagNamespacePage == false)
            {
                FuncInfo info = new FuncInfo();
                info.RefId = refid;
                info.FunctionName = null;
                info.NamespaceName = namespaceName;
                if (!CheckFunctionInfo(info))
                {
                    stringList.Add(info);
                }
            }
            return true;
        }

        private static bool FunctionHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string functionName)
        {
            if (cmode == DOXYGEN_XML.CompoundMode.COMPOUND_FILE)
            {
                return true;
            }

            FuncInfo info = new FuncInfo();
            if (CmdlineOption.FlagNamespacePage == true)
            {
                info.RefId = refid;
            }
            else
            {
                info.RefId = Common.UtilDoxygen.CutDoxygenId(member_refid);
            }
            info.FunctionName = functionName;
            info.NamespaceName = namespaceName;
            if (!CheckFunctionInfo(info))
            {
                stringList.Add(info);
            }
            return true;
        }

        private static bool StructHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string structName)
        {
            if (cmode == DOXYGEN_XML.CompoundMode.COMPOUND_FILE)
            {
                return true;
            }

            FuncInfo info;
            info.RefId = refid;
            info.FunctionName = structName;
            info.NamespaceName = null;
            if (!CheckFunctionInfo(info))
            {
                stringList.Add(info);
            }
            return true;
        }

        private static bool DefineHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string definitionName)
        {
            FuncInfo info = new FuncInfo();
            if (CmdlineOption.FlagNamespacePage == true)
            {
                info.RefId = refid;
            }
            else
            {
                info.RefId = Common.UtilDoxygen.CutDoxygenId(member_refid);
            }
            info.FunctionName = definitionName;
            info.NamespaceName = null;
            if (!CheckFunctionInfo(info))
            {
                stringList.Add(info);
            }
            return true;
        }

        private static bool VariableHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string variableName)
        {
            if (cmode == DOXYGEN_XML.CompoundMode.COMPOUND_FILE)
            {
                return true;
            }

            FuncInfo info = new FuncInfo();
            if (CmdlineOption.FlagNamespacePage == true)
            {
                info.RefId = refid;
            }
            else
            {
                info.RefId = Common.UtilDoxygen.CutDoxygenId(member_refid);
            }
            info.FunctionName = variableName;
            info.NamespaceName = namespaceName;
            if (!CheckFunctionInfo(info))
            {
                stringList.Add(info);
            }
            return true;
        }

        private static bool EnumHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string enumName)
        {
            if (cmode == DOXYGEN_XML.CompoundMode.COMPOUND_FILE)
            {
                return true;
            }

            FuncInfo info = new FuncInfo();
            if (CmdlineOption.FlagNamespacePage == true)
            {
                info.RefId = refid;
            }
            else
            {
                info.RefId = Common.UtilDoxygen.CutDoxygenId(member_refid);
            }
            info.FunctionName = enumName;
            info.NamespaceName = namespaceName;
            if (!CheckFunctionInfo(info))
            {
                stringList.Add(info);
            }
            return true;
        }

        private static bool EnumValueHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string enumValueName)
        {
            if (cmode == DOXYGEN_XML.CompoundMode.COMPOUND_FILE)
            {
                return true;
            }

            FuncInfo info = new FuncInfo();
            if (CmdlineOption.FlagNamespacePage == true)
            {
                info.RefId = refid;
            }
            else
            {
                info.RefId = Common.UtilDoxygen.CutDoxygenId(member_refid);
            }
            info.FunctionName = enumValueName;
            info.NamespaceName = namespaceName;
            if (!CheckFunctionInfo(info))
            {
                stringList.Add(info);
            }
            return true;
        }

        private static bool FileHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string enumValueName)
        {
            return true;
        }

        /// <summary>
        /// doxygen生成HTMLファイルに各種metaタグを追記します
        /// </summary>
        public static void Write()
        {
            if (doxygenXmlPath == null)
            {
                return;
            }

            /* Microsoft.Help.F1 のmetaタグを追加する */
            WriteF1MetaTag();

            /* Microsoft.Help.Category のmetaタグを追加する */
            WriteCategoryMetaTag();
        }

        /// <summary>
        /// VsHelpでは &nbsp; が使用できないため書き換えます
        /// </summary>
        public static void ConvertNbsp()
        {
            if (doxygenHtmlPath == null)
            {
                return;
            }

            string line = null;
            string[] files = Common.UtilFs.GetDoxygenHtmlFiles();

            foreach (string filename in files)
            {
                string filepath = doxygenHtmlPath + Path.DirectorySeparatorChar + filename;
                string fileData;
                FILEIO.GetData(out fileData, filepath);
                using (StringReader sr = new StringReader(fileData))
                using (StringWriter sw = new StringWriter())
                {
                    sw.NewLine = "\n";
                    // "&#160;"を見つけたら空白に置き換えて書き込む
                    while (sr.Peek() > -1)
                    {
                        line = sr.ReadLine();
                        System.Text.StringBuilder sb = new System.Text.StringBuilder(line);
                        sb.Replace("&#160;", " ");
                        sw.WriteLine(sb.ToString());
                    }
                    string data = sw.ToString();
                    FILEIO.SetData(ref data, filepath);
                }
            }
        }

        private static bool WriteF1MetaTag()
        {
            string line = null;

            /* Microsoft.Help.F1 のmetaタグを追加する */
            foreach (FuncInfo info in stringList)
            {
                // ファイルをオープンする
                string filename = info.RefId + ".html";
                string filepath = doxygenHtmlPath + "\\" + filename;

                if (!Common.UtilFs.CheckDoxygenHtmlFileExist(filename))
                {
                    // htmlファイルがない場合はスキップする
                    continue;
                }

                if (info.NamespaceName == null & info.FunctionName == null)
                {
                    continue;
                }

                string fileData;
                FILEIO.GetData(out fileData, filepath);
                using (StringReader sr = new StringReader(fileData))
                using (StringWriter sw = new StringWriter())
                {
                    sw.NewLine = "\n";
                    // metaの書けそうな場所を探す
                    //   行毎に読み込み </head>の行の直前まで書き込むようにする
                    while (sr.Peek() > -1)
                    {
                        line = sr.ReadLine();
                        if (line.StartsWith("</head>"))
                        {
                            break;
                        }
                        sw.WriteLine(line);
                    }
                    // metaを書く
                    string metastring;
                    if (info.NamespaceName == null && info.FunctionName != null)
                    {
                        metastring = "<meta name=\"Microsoft.Help.F1\" content=\"" + ConvertSpecialCharacters(info.FunctionName) + "\" />";
                    }
                    else if (info.NamespaceName != null && info.FunctionName == null)
                    {
                        metastring = "<meta name=\"Microsoft.Help.F1\" content=\"" + ConvertSpecialCharacters(info.NamespaceName) + "\" />";
                    }
                    else if (info.NamespaceName != null && info.FunctionName != null)
                    {
                        metastring = "<meta name=\"Microsoft.Help.F1\" content=\"" + ConvertSpecialCharacters(info.NamespaceName) + "::" + ConvertSpecialCharacters(info.FunctionName) + "\" />";
                    }
                    else
                    {
                        metastring = string.Empty;
                    }
                    sw.WriteLine(metastring);
                    // </head>以降の行も書き込む
                    sw.WriteLine(line);
                    while (sr.Peek() > -1)
                    {
                        line = sr.ReadLine();
                        sw.WriteLine(line);
                    }
                    string data = sw.ToString();
                    FILEIO.SetData(ref data, filepath);
                }
            }
            return true;
        }

        private static bool WriteCategoryMetaTag()
        {
            string line = null;

            // 重複項目は1つを除いて除外する
            List<string> refIdList = new List<string>();
            foreach (FuncInfo info in stringList)
            {
                if (!refIdList.Contains(info.RefId))
                {
                    refIdList.Add(info.RefId);
                }
            }
            /* Microsoft.Help.Category のmetaタグを追加する */
            foreach (string refid in refIdList)
            {
                string filename = refid + ".html";
                string filepath = doxygenHtmlPath + "\\" + filename;

                if (!Common.UtilFs.CheckDoxygenHtmlFileExist(filename))
                {
                    // htmlファイルがない場合はスキップする
                    continue;
                }

                string fileData;
                FILEIO.GetData(out fileData, filepath);
                using (StringReader sr = new StringReader(fileData))
                using (StringWriter sw = new StringWriter())
                {
                    sw.NewLine = "\n";
                    // metaの書けそうな場所を探す
                    //   行毎に読み込み </head>の行の直前まで書き込むようにする
                    while (sr.Peek() > -1)
                    {
                        line = sr.ReadLine();
                        if (line.StartsWith("</head>"))
                        {
                            break;
                        }
                        sw.WriteLine(line);
                    }

                    // metaを書く
                    string metastring = "<meta name=\"Microsoft.Help.Category\" content=\"DevLang:C++\" />";
                    sw.WriteLine(metastring);
                    // </head>以降の行も書き込む
                    sw.WriteLine(line);
                    while (sr.Peek() > -1)
                    {
                        line = sr.ReadLine();
                        sw.WriteLine(line);
                    }
                    string data = sw.ToString();
                    FILEIO.SetData(ref data, filepath);
                }
            }

            return true;
        }

        private static void ParseOverloadFunctions()
        {
            string listfile = doxygenXmlPath + "\\overloadlist.xml";
            Encoding encode = Encoding.GetEncoding("UTF-8");
            if (!File.Exists(listfile))
            {
                return;
            }

            List<string> refidList = new List<string>();
            {
                string fileData;
                FILEIO.GetData(out fileData, listfile);
                using (StringReader sr = new StringReader(fileData))
                using (XmlTextReader reader = new XmlTextReader(sr))
                {
                    while (reader.Read())
                    {
                        if (reader.NodeType != XmlNodeType.Element)
                        {
                            continue;
                        }

                        // 要素 innerpage を探す
                        // MEMO: それより上の階層の解析は省略しています
                        if (reader.LocalName.Equals("innerpage"))
                        {
                            // refidを保存する
                            for (int i = 0; i < reader.AttributeCount; i++)
                            {
                                reader.MoveToAttribute(i);
                                if (reader.Name == "refid")
                                {
                                    refidList.Add(reader.Value);
                                }
                            }
                        }
                        reader.MoveToElement();
                    }
                }
            }

            foreach (string childname in refidList)
            {
                string childXmlPath = doxygenXmlPath + "\\" + childname + ".xml";
                if (!File.Exists(childXmlPath))
                {
                    LOG.LogLine(LOG.Level.LOG_ERROR, "xml file not found : {0}", childXmlPath);
                    continue;
                }
                string fileData;
                FILEIO.GetData(out fileData, childXmlPath);
                using (StringReader sr = new StringReader(fileData))
                using (XmlTextReader reader = new XmlTextReader(sr))
                {
                    while (reader.Read())
                    {
                        if (reader.NodeType != XmlNodeType.Element)
                        {
                            continue;
                        }
                        // 要素 compoundname を探す
                        if (reader.LocalName.Equals("compoundname"))
                        {
                            // オーバーロードAPI名を取得する
                            string compoundname = reader.ReadString();
                            if (compoundname.Contains("overloadc_"))
                            {
                                string overloadApiName = compoundname.Substring(10).Replace("_", "::");
                                FuncInfo info = new FuncInfo();
                                info.FunctionName = overloadApiName;
                                info.RefId = childname;
                                overloadList.Add(info);
                            }
                            else
                            {
                                LOG.LogLine(LOG.Level.LOG_ERROR, "illigal compound name : {0}", compoundname);
                            }
                        }
                        reader.MoveToElement();
                    }
                }
            }
        }

        private static void AdjustOverloadApiList()
        {
            List<FuncInfo> tmpList = new List<FuncInfo>();
            foreach (FuncInfo apiInfo in stringList)
            {
                bool isOverload = false;
                string apiFullName;
                if (apiInfo.NamespaceName == null)
                {
                    apiFullName = apiInfo.FunctionName;
                }
                else
                {
                    apiFullName = apiInfo.NamespaceName + "::" + apiInfo.FunctionName;
                }
                foreach (FuncInfo overloadInfo in overloadList)
                {
                    // overloadInfo.namespaceName は常にnull
                    if (overloadInfo.FunctionName == apiFullName)
                    {
                        isOverload = true;
                        break;
                    }
                }
                if (!isOverload)
                {
                    tmpList.Add(apiInfo);
                }
            }
            stringList.Clear();
            stringList = new List<FuncInfo>(tmpList);
            var list = stringList.Concat<FuncInfo>(overloadList);
            stringList = list.ToList<FuncInfo>();
        }

        private static string ConvertSpecialCharacters(string line)
        {
            System.Text.StringBuilder sb = new System.Text.StringBuilder(line);
            sb.Replace("©", "&#169;");
            sb.Replace("\"", "&#34;");
            sb.Replace("&", "&#38;");
            sb.Replace("<", "&#60;");
            sb.Replace(">", "&#62;");
            return sb.ToString();
        }
    }
}
