﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Windows.Markup.Localizer;

namespace Nintendo.DotNetLocalizer
{
    internal class LocalizationResourceEnumerator
    {
        public LocalizationResourceEnumerator()
        {
        }

        public IEnumerable<LocalizationItem> EnumerateLocalizeItems(IEnumerable<FileInfo> asmFiles)
        {
            return asmFiles.SelectMany(asmFile => WriteString(asmFile));
        }

        /// <summary>
        /// 文字列出力。
        /// </summary>
        private IEnumerable<LocalizationItem> WriteString(FileInfo asmFile)
        {
            var assembly = Assembly.Load(Path.GetFileNameWithoutExtension(asmFile.Name));
            foreach (var resourceFileName in AssemblyUtility.EnumerateResourceName(assembly))
            {
                // 全エントリを順に処理する
                foreach (var entry in AssemblyUtility.EnumerateResouces(assembly.GetManifestResourceStream(resourceFileName)))
                {
                    var resName = Utility.TrimResourcesExt(resourceFileName);
                    string strValue;
                    if (null != (strValue = entry.Value as string))
                    {
                        // 翻訳対象のキーなら
                        if (!DllGenerator.IsExcludeResourceKey(entry.Key))
                        {
                            var japanese = Utility.CRLFToLF(strValue);
                            // 翻訳対象文字列ならリストに格納
                            if (CheckLocalizableString(japanese))
                            {
                                yield return new LocalizationItem(Path.GetFileNameWithoutExtension(asmFile.Name), resName, entry.Key, japanese);
                            }
                        }
                    }
                    else if (DllGenerator.IsResourceEntryBamlStream(entry.Key, entry.Value))
                    {
                        var enumBamlLocItems = WriteStringBaml_(asmFile, resName, entry.Key, (Stream)entry.Value);
                        foreach (var item in enumBamlLocItems)
                        {
                            yield return item;
                        }
                    }
                }

                Output.DebugWrite(" {0} - ", resourceFileName);
            }
        }

        /// <summary>
        /// 文字列出力(baml)。
        /// </summary>
        private IEnumerable<LocalizationItem> WriteStringBaml_(FileInfo asmFile, string resName, string bamlKeyName, Stream baml)
        {
            // BamlLocalizer の中で、「pack:」スキーマを使った Uri コンストラクタが
            // 失敗しないようにするため PackUriHelper のスタティックコンストラクタが
            // 呼ばれるようにしておきます。
            var uriSchemePack = System.IO.Packaging.PackUriHelper.UriSchemePack;

            BamlLocalizer bamlLocalizer = null;
            try
            {
                bamlLocalizer = new BamlLocalizer(baml);
            }
            catch (Exception e)
            {
                Output.Error(e.Message);
                Utility.TestError(
                    false,
                    "関連するプロジェクトがビルドされているかどうか、" +
                    "確認してください。");
            }

            return
                from entry in bamlLocalizer.ExtractResources().OfType<DictionaryEntry>()
                where null != entry.Value
                let value = (BamlLocalizableResource)entry.Value
                where CheckLocalizableString(value.Content)
                select new LocalizationItem(
                    Path.GetFileNameWithoutExtension(asmFile.Name),
                    string.Format("{0}:{1}", resName, bamlKeyName),
                    AssemblyUtility.GetKeyString((BamlLocalizableResourceKey)entry.Key),
                    value.Content);
        }

        /// <summary>
        /// 翻訳対象文字列か。
        /// </summary>
        private static bool CheckLocalizableString(string input)
        {
            // 日本語文字が含まれるかどうか
            foreach (char ch in input)
            {
                if (ch >= 0x0080)
                {
                    return true;
                }
            }
            return false;
        }
    }
}
