﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.IO;
using YamlDotNet;
using YamlDotNet.Serialization;

namespace CsScripts
{
    public static class TargetInitializationUtility
    {
        public class ImageType
        {
            public string Name { get; set; }
            public string KeyType { get; set; }
            public string ConnectionType { get; set; }
            public string SignedType { get; set; }
            public string BootType { get; set; }
            public string StorageSize { get; set; }
            public string DeveloperType { get; set; }
            public string BuildType { get; set; }

            public ImageType(string buildType, string[] parameterArray)
            {
                Name = MakeDefaultName(buildType, parameterArray);
                KeyType = parameterArray[0];
                ConnectionType = parameterArray[1];
                SignedType = parameterArray[2];
                BootType = parameterArray[3];
                StorageSize = parameterArray[4];
                DeveloperType = parameterArray[5];
                BuildType = buildType;
            }

            public ImageType(string name, string buildType, string[] parameterArray)
            {
                Name = name;
                KeyType = parameterArray[0];
                ConnectionType = parameterArray[1];
                SignedType = parameterArray[2];
                BootType = parameterArray[3];
                StorageSize = parameterArray[4];
                DeveloperType = parameterArray[5];
                BuildType = buildType;
            }

            public static ImageType MakeFromImageTypeName(string name, string imageTypeName)
            {
                var match = Regex.Match(imageTypeName, "NX(-([^-]+)){6}-([^-]+)");

                if (!match.Success)
                {
                    var message = string.Format("Unsupported imageTypeName: {0}", imageTypeName);
                    Console.WriteLine(message);
                    throw new Exception(message);
                }

                var parameters = match.Groups[2].Captures.Cast<Capture>().Select(capture => capture.Value).ToArray();
                var buildType = match.Groups[3].Value;

                if(name == "Auto")
                {
                    return new ImageType(buildType, parameters);
                }
                else
                {
                    return new ImageType(name, buildType, parameters);
                }
            }

            public static List<string> PopularDeveloperType = new List<string> {
                "Internal",
                "Public",
                "ProdMode",
                "ProdModeSdLog"
            };


            public bool IsPopular
            {
                get
                {
                    return PopularDeveloperType.Contains(this.DeveloperType);
                }
            }

            public static string MakeDefaultName(string buildType, string[] parameterArray)
            {
                var p = parameterArray;
                return string.Format("Initialize-NX-{0}-{1}-{2}-{3}-{4}-{5}-{6}", p[0], p[1], p[2], p[3], p[4], p[5], buildType);
            }

            public string ScriptName
            {
                get
                {
                    return "Initialize-Nx-Raw.ps1";
                }
            }

            public string KeyTypeForScript
            {
                get
                {
                    if (ConnectionType == "Hb")
                    {
                        return "Auto";
                    }
                    else
                    {
                        return KeyType;
                    }
                }
            }

            public string BatchFileName
            {
                get
                {
                    return this.Name + ".bat";
                }
            }

            public static ImageType[] MakeFromYamlFile(FileInfo yamlFile)
            {
                return ImageTypeDefinitions.MakeFromYamlFile(yamlFile).Definitions.Select(def => def.ImageType).ToArray();
            }

            public static ImageType[] MakeFromNactFile(FileInfo nactFile)
            {
                var imageTypes = new List<ImageType>();

                foreach (var line in File.ReadAllLines(nactFile.FullName))
                {
                    var match = Regex.Match(line, ".*initializedNx\\((\\\"([^\\\"]*)\\\",?\\s*){6}.*\\).*");

                    if (match.Success)
                    {
                        var c = match.Groups[2].Captures.Cast<Capture>().Select(capture => capture.Value).ToArray();

                        foreach (var buildType in BuildTypes)
                        {
                            var imageType = new ImageType(buildType, c);

                            if (imageType.IsPopular)
                            {
                                imageTypes.Add(imageType);
                            }
                        }
                    }
                }

                return imageTypes.GroupBy(x => x.Name, (key, group) => group.First()).ToArray();
            }
        }

        public class ImageTypeDefinitions
        {
            public class ImageTypeDefinition
            {
                public string Name { get; set; }

                [YamlAlias("ImageType")]
                public string ImageTypeName { get; set; }

                [YamlIgnore]
                public ImageType ImageType
                {
                    get
                    {
                        return ImageType.MakeFromImageTypeName(Name, ImageTypeName);
                    }
                }
            }

            public ImageTypeDefinition[] Definitions { get; set; }

            public static ImageTypeDefinitions MakeFromYamlFile(FileInfo fileInfo)
            {
                return LoadYaml<ImageTypeDefinitions>(fileInfo);
            }
        }

        public class GeneratingContext
        {
            public string SdkRoot { get; set; }
            public string Category { get; set; }
            public string Source { get; set; }
            public Func<FileInfo, ImageType[]> ImageTypeGenerator { get; set; }
            public FileInfo SourceFile
            {
                get
                {
                    return new FileInfo(Path.Combine(SdkRoot, Source));
                }
            }
            public FileInfo DestinationNactFile
            {
                get
                {
                    return new FileInfo(Path.Combine(SdkRoot, DestinationNactFileRoot, Category, "!.nact"));
                }
            }
            public DirectoryInfo DestinationLabelFileDirectory
            {
                get
                {
                    return new DirectoryInfo(Path.Combine(SdkRoot, DestinationNactFileRoot, Category));
                }
            }
            public string LabelFileDirectory
            {
                get
                {
                    return Path.Combine(DestinationNactFileRoot, Category);
                }
            }
            public DirectoryInfo DestinationScriptDirectory
            {
                get
                {
                    return new DirectoryInfo(Path.Combine(SdkRoot, DestinationScriptFileRoot, Category));
                }
            }

            public ImageType[] MakeImageTypes()
            {
                return ImageTypeGenerator(SourceFile);
            }

            public static string DestinationNactFileRoot = "Integrate/Sources/InitializeTarget/NX";
            public static string DestinationScriptFileRoot = "Integrate/Scripts/NX/Initialize";
        }

        public static T LoadYaml<T>(string text)
        {
            var deserializer = new Deserializer();

            using (var textReader = new StringReader(text))
            {
                return deserializer.Deserialize<T>(textReader);
            }
        }

        public static T LoadYaml<T>(FileInfo yamlFile)
        {
            var yamlText =  File.ReadAllText(yamlFile.FullName);

            return LoadYaml<T>(yamlText);
        }

        public static string MakeHeader(string lineComment, string generationScript, string sourcePath)
        {
            var template = @"{0}
{0} generated by:
{0}     {1}
{0}
{0} definition source: {2}
{0}
";
            return string.Format(template, lineComment, generationScript, sourcePath);
        }

        private static void GenerateInitializeScripts(DirectoryInfo destinationDirectory, string sourcePath, string labelDirectory, ImageType[] definitions)
        {
            foreach (var imageType in definitions)
            {
                var scriptPath = new FileInfo(Path.Combine(destinationDirectory.FullName, imageType.BatchFileName));
                CreateInitializeTargetScript(scriptPath, labelDirectory, imageType, "../", sourcePath);
            }
        }

        private static void GenerateLabelFiles(DirectoryInfo destinationDirectory, ImageType[] definitions)
        {
            foreach (var imageType in definitions)
            {
                var labelFilePath = new FileInfo(Path.Combine(destinationDirectory.FullName, imageType.Name));

                if (!labelFilePath.Exists)
                {
                    Console.WriteLine("Create: {0}", labelFilePath.FullName);
                    labelFilePath.Create().Close();
                }
            }
        }

        private static void GenerateNactFile(string sourcePath, FileInfo destinationRuleFile, ImageType[] definitions)
        {
            using (var writeStream = destinationRuleFile.Create())
            using (var writer = new StreamWriter(writeStream, Encoding.UTF8))
            {
                writer.WriteLine(MakeHeader("#", GeneratingScriptPath, sourcePath));
                foreach (var imageType in definitions)
                {
                    writer.WriteLine("NxInitialImagePseudoPackage");
                    writer.WriteLine("{");
                    writer.WriteLine("    @override LabelFile = f\"{0}\";", imageType.Name);
                    writer.WriteLine("    @override Config = MakeConfig(\"{0}\", \"{1}\", \"{2}\", \"{3}\", \"{4}\", \"{5}\", \"{6}\");",
                        imageType.KeyType,
                        imageType.ConnectionType,
                        imageType.SignedType,
                        imageType.BootType,
                        imageType.StorageSize,
                        imageType.DeveloperType,
                        imageType.BuildType);
                    writer.WriteLine("}");
                    writer.WriteLine("");
                }
            }
        }

        private static void CreateInitializeTargetScript(FileInfo scriptPath, string labelDirectory, ImageType imageType, string relativePathForScript, string sourcePath)
        {
            using (var writeStream = scriptPath.Create())
            using (var writer = new StreamWriter(writeStream, Encoding.UTF8))
            {
                writer.WriteLine(MakeHeader("::", GeneratingScriptPath, sourcePath));

                var template = @"@echo on

%~d0
cd %~p0

echo . | powershell -Version 2.0 -InputFormat None -ExecutionPolicy RemoteSigned ^
    {0}{1} ^
    -KeyType {2} ^
    -ConnectionType {3} ^
    -SignedType {4} ^
    -BootType {5} ^
    -StorageSize {6} ^
    -DeveloperType {7} ^
    -BuildType {8} ^
    -LabelFile {9}/{10} ^
    %*

if %ERRORLEVEL% neq 0 exit /b %ERRORLEVEL%

exit /b %ERRORLEVEL%
";

                writer.WriteLine(template,
                    relativePathForScript,
                    imageType.ScriptName,
                    imageType.KeyTypeForScript,
                    imageType.ConnectionType,
                    imageType.SignedType,
                    imageType.BootType,
                    imageType.StorageSize,
                    imageType.DeveloperType,
                    imageType.BuildType,
                    labelDirectory,
                    imageType.Name);
            }
        }

        public static readonly string[] BuildTypes = new string[] { "Develop", "Release" };
        public static string GeneratingScriptPath = "Integrate/Scripts/NX/Utility/Update-InitialImageDefinitions.bat";

        public static void GenerateFiles(GeneratingContext context)
        {
            var imageTypes = context.MakeImageTypes();
            GenerateNactFile(context.Source, context.DestinationNactFile, imageTypes);
            GenerateLabelFiles(context.DestinationLabelFileDirectory, imageTypes);
            GenerateInitializeScripts(context.DestinationScriptDirectory, context.Source, context.LabelFileDirectory, imageTypes);
        }

        public static void GenerateBuildRulesAndScriptsForInitialImage(string sdkRoot)
        {
            GenerateFiles(new GeneratingContext()
            {
                SdkRoot = sdkRoot,
                Category = "Basic",
                Source = "Integrate/Sources/InitializeTarget/NX/Definitions/BasicInitialImageDefinitions.yaml",
                ImageTypeGenerator = source => ImageTypeDefinitions.MakeFromYamlFile(source).Definitions.Select(def => def.ImageType).ToArray()
            });

            GenerateFiles(new GeneratingContext()
            {
                SdkRoot = sdkRoot,
                Category = "Misc",
                Source = "Integrate/Sources/InitializeTarget/NX/Definitions/MiscInitialImageDefinitions.yaml",
                ImageTypeGenerator = source => ImageTypeDefinitions.MakeFromYamlFile(source).Definitions.Select(def => def.ImageType).ToArray()
            });

            GenerateFiles(new GeneratingContext()
            {
                SdkRoot = sdkRoot,
                Category = "Detail",
                Source = "Programs/Eris/Sources/InitialImages/InitializedNx/!.nact",
                ImageTypeGenerator = source => ImageType.MakeFromNactFile(source)
            });
        }
    }
}
