﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace PushChecker
{
    internal class RefCasingConflictDetector
    {
        private static readonly char[] Separator = new char[] { '/' };

        private readonly string[] _remoteRefNames;

        /// <summary>
        /// 指定した remote refs を使用して検査を行うように構築します。
        /// </summary>
        public RefCasingConflictDetector(IEnumerable<string> remoteRefNames)
        {
            this._remoteRefNames = remoteRefNames.ToArray();
        }

        /// <summary>
        /// remote に生成しようとしている ref の名前が、既存の remote refs の名前と衝突しないかどうかを検査します。
        /// 衝突するとは、大文字小文字だけが異なる名前を持つということです。 (例: feature/A と feature/a)
        /// </summary>
        public RefCasingConflictDetectorResult Detect(string newRefName)
        {
            var conflictingRemoteRefName = _remoteRefNames.FirstOrDefault(x => Conflicts(newRefName, x));

            if (conflictingRemoteRefName == null)
            {
                return RefCasingConflictDetectorResult.CreateNoConflict();
            }
            else
            {
                return RefCasingConflictDetectorResult.CreateConflict(conflictingRemoteRefName);
            }
        }

        // 1つの remote ref と衝突するかどうかを返します
        private bool Conflicts(string newRefName, string remoteRefNames)
        {
            var newParts = Split(newRefName);
            var remoteParts = Split(remoteRefNames);

            // ルートから順番に各部分を比較
            for (int i = 0; i < Math.Min(newParts.Length, remoteParts.Length); i++)
            {
                var n = newParts[i];
                var r = remoteParts[i];

                // 大文字小文字を区別しない比較でも一致しないなら、別の名前なので OK
                if (StringComparer.OrdinalIgnoreCase.Compare(n, r) != 0)
                {
                    return false;
                }
                // 大文字小文字だけが異なるなら衝突、 NG
                else if (StringComparer.Ordinal.Compare(n, r) != 0)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// ref を、'/' で区切られた部分に分割する
        /// </summary>
        private string[] Split(string refName) => refName.Split(Separator);
    }

    internal class RefCasingConflictDetectorResult
    {
        // HACK: 同値比較を Tuple に委譲する
        private Tuple<bool, string> _value;

        public bool HasConflict => _value.Item1;
        public string ConflictingRemoteRef => _value.Item2;

        private RefCasingConflictDetectorResult(bool hasConflict, string conflictingRemoteRef)
        {
            this._value = Tuple.Create(hasConflict, conflictingRemoteRef);
        }

        public static RefCasingConflictDetectorResult CreateNoConflict() =>
            new RefCasingConflictDetectorResult(false, null);

        public static RefCasingConflictDetectorResult CreateConflict(string conflictingRemoteRef) =>
            new RefCasingConflictDetectorResult(true, conflictingRemoteRef);

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }

            var other = obj as RefCasingConflictDetectorResult;
            return EqualityComparer<Tuple<bool, string>>.Default.Equals(this._value, other._value);
        }

        public override string ToString() => _value.ToString();
        public override int GetHashCode() => _value.GetHashCode();
        public static bool operator ==(RefCasingConflictDetectorResult lhs, RefCasingConflictDetectorResult rhs) => EqualityComparer<Tuple<bool, string>>.Default.Equals(lhs._value, rhs._value);
        public static bool operator !=(RefCasingConflictDetectorResult lhs, RefCasingConflictDetectorResult rhs) => lhs != rhs;
    }
}
