﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.IO;
using CodingCheckerUtil;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace PushChecker
{
    internal class Program
    {
        public static void Main(string[] args)
        {
            // コマンドライン引数を解析
            var parseResult = ParseCommandLine(args);
            if (parseResult.Params == null)
            {
                Environment.Exit(parseResult.ExitCode);
                return;
            }

            var @params = parseResult.Params;

            // git ls-remote の出力を得てパースする
            IReadOnlyCollection<RefInfo> remoteRefs;
            try
            {
                remoteRefs = GetRemoteRefs(@params.GitPath, @params.Remote);
            }
            catch (GitExecuterException e)
            {
                Console.Error.WriteLine(e.Message);
                Environment.Exit(1);
                return;
            }

            // 標準入力に与えられた各 push 対象について、大文字小文字だけが異なる名前で push していないか検査する
            var pushInfos = Utils.ParsePrePushInput(Console.In.ReadToEnd());
            var detector = new RefCasingConflictDetector(remoteRefs.Select(x => x.Name));
            bool hasError = false;
            foreach (var pushInfo in pushInfos)
            {
                var result = detector.Detect(pushInfo.DestinationRef.Name);
                if (result.HasConflict)
                {
                    Console.Error.WriteLine($"error: Destination ref '{pushInfo.DestinationRef.Name}' differs in casing with a remote ref '{result.ConflictingRemoteRef}'.");
                    hasError = true;
                    continue;
                }
            }

            if (hasError)
            {
                Console.Error.WriteLine("Fix casing of the destination ref so it matches the remote ref.");
                Environment.Exit(1);
                return;
            }

            Environment.Exit(0);
        }

        private static IReadOnlyCollection<RefInfo> GetRemoteRefs(string gitPath, string remote)
        {
            var gitExecuter = new GitExecuter(gitPath);
            return gitExecuter.LsRemote(remote);
        }

        private static ParseCommandLineResult ParseCommandLine(string[] args)
        {
            var s = new CommandLineParserSettings()
            {
                ApplicationDescription = "Validates what is going to be git-push'ed.",
            };

            try
            {
                PushCheckerParams @params;
                if (!new CommandLineParser(s).ParseArgs(args, out @params))
                {
                    return new ParseCommandLineResult(0);
                }
                return new ParseCommandLineResult(@params);
            }
            catch (ArgumentException)
            {
                return new ParseCommandLineResult(1);
            }
        }

        private class ParseCommandLineResult
        {
            public PushCheckerParams Params { get; }
            public int ExitCode { get; }

            public ParseCommandLineResult(PushCheckerParams @params)
            {
                this.Params = @params;
            }

            public ParseCommandLineResult(int exitCode)
            {
                this.Params = null;
                this.ExitCode = exitCode;
            }
        }
    }
}
