﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Reflection;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Runtime.CompilerServices;
using System.Diagnostics.Contracts;

namespace CommitLogChecker
{
    using CheckResult = Tuple<bool, int>;

    internal class CommitLogChecker
    {
        private static Regex RegexException = new Regex(@"(\Afixup!)|([Ww][Ii][Pp]:)");
        private const string ModulePattern = @"[\w\-]+?";
        private const string JiraKeySubstituteString = "$(JIRA-KEY)";   // 課題キーの代わりに記述する文字列（課題キーに置換される）

        private List<string> m_JiraKeys { get; }

        internal CommitLogChecker(IEnumerable<string> jiraKeys)
        {
            m_JiraKeys = jiraKeys.ToList();
        }

        internal bool CheckCommitLog(string commitLog)
        {
            // 行で分割します。
            string[] logSplitedLines = SplitIntoLine(commitLog);

            const string CommentHead = "#"; // コメント行の先頭文字
            int lineIndex = 0; // 行のインデックス、ただしコメントを含まない。
            bool checkedFlag = false; // チェックが行われた場合 true

            foreach (string logLine in logSplitedLines)
            {
                if (!logLine.StartsWith(CommentHead)) // コメントではない行のみチェックを行う
                {
                    if (!CheckCommitLogByLine(logLine, lineIndex, m_JiraKeys))
                    {
                        return false;
                    }
                    checkedFlag = true;
                    lineIndex++;
                }
            }

            if (!checkedFlag) // コメント行のみでチェックが行われていない場合
            {
                Console.Error.WriteLine(@"CommitLogCheck: Error: This commit log is empty.");
                return false;
            }
            return true;
        }

        internal bool CheckExceptionCommit(string commitLog)
        {
            var m = RegexException.Match(commitLog);
            return m.Success;
        }

        // 各チェックを行ごとに分ける
        private static bool CheckCommitLogByLine(string commitLogLine, int lineNumber, List<string> jiraKeys)
        {
            Contract.Requires(lineNumber >= 0);

            if (!CheckJiraKeySubstituteReplaced(commitLogLine))
            {
                return false;
            }

            if (lineNumber == 0) // 一行目のチェック
            {
                if (!CheckSummaryLine(commitLogLine, jiraKeys))
                {
                    return false;
                }
                return true;
            }
            else if (lineNumber == 1) // 二行目のチェック
            {
                // 空行チェック
                if (commitLogLine != string.Empty)
                {
                    Console.Error.WriteLine(@"CommitLogCheck: Error: The second line must be empty.");
                    return false;
                }
                return true;
            }
            return true;
        }

        // $(JIRA-KEY) が残っていないことを確認する
        private static bool CheckJiraKeySubstituteReplaced(string commitLog)
        {
            if (commitLog.Contains(JiraKeySubstituteString))
            {
                Console.Error.WriteLine(@"CommitLogCheck: Error: $(JIRA-KEY) cannot be replaced; the branch name does not contain a valid JIRA-KEY.");
                return false;
            }

            return true;
        }

        private static bool CheckSummaryLine(string commitLogLine, List<string> jiraKeys)
        {
            string exampleCommitLog = "(JIRAKEY-xxx) This is sample commit.";
            string errorFormatStr = "CommitLogCheck: Error: {0} is wrong \n {1} \n Example :\n " + exampleCommitLog;

            // JIRAKEY チェック
            string CheckJiraKeyPattern = jiraKeys.Count != 0 ? @"\G\((" + string.Join("|", jiraKeys) + @")\-\d+\)" : @"\G\(\S+\-\d+\)";
            CheckResult result = CheckFormat(commitLogLine, 0, CheckJiraKeyPattern);
            if (!result.Item1)
            {
                Console.Error.WriteLine(errorFormatStr, "JIRA KEY", "You must write a right JIRA KEY at the beginning of a commit log.");
                Console.Error.WriteLine("JIRA KEY LIST:" + string.Join(", ", jiraKeys));

                return false;
            }

            return true;
        }

        // 行分割をし、分割された文字列を返します。
        private static string[] SplitIntoLine(string commitLog)
        {
            char[] lineFeedCode = new char[] { '\n' }; // Splitの仕様上、配列にしている。

            // 改行コードを全て\nに変換した後に分割する。末尾改行は分割前に取り除く
            string replacedLog = commitLog.Replace("\r\n", "\n").Replace("\r", "\n");
            replacedLog = Regex.Replace(replacedLog, @"\n$", string.Empty, RegexOptions.Singleline | RegexOptions.RightToLeft);

            return replacedLog.Split(lineFeedCode, StringSplitOptions.None);
        }

        // コミットログと文字の位置及びパターンを設定して
        // CheckResult.Item1に Match 結果、Item2 に Match 終了文字位置
        private static CheckResult CheckFormat(string commitLog, int stringPos, string pattern)
        {
            Regex regex = new Regex(pattern, RegexOptions.Singleline);
            Match match = regex.Match(commitLog, stringPos);

            return new CheckResult(match.Success, match.Index + match.Length);
        }
    }
}
