﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Reflection;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Runtime.CompilerServices;


[assembly: InternalsVisibleTo("CodingCheckerUtilTest")]

namespace CodingCheckerUtil
{
    [Flags]
    public enum CheckOptions
    {
        None = 0,

        // Byte Order Mark が存在することをチェックする
        CheckBOM = 1,

        // 機械的なテキストの整形を行う
        AutoFormatContents = 2,

        All = CheckBOM | AutoFormatContents
    }

    public class TargetExtensionListParser
    {
        internal Dictionary<string, CheckOptions> TargetFileCheckOptionsForDebug => m_TargetFileCheckOptions;

        private Dictionary<string, CheckOptions> m_TargetFileCheckOptions { get; }

        public TargetExtensionListParser(string targetExtensionListFile)
        {
            if (string.IsNullOrEmpty(targetExtensionListFile))
            {
                m_TargetFileCheckOptions = null;
            }
            else
            {
                try
                {
                    // コード整形対象の拡張子指定ファイルを読み込み
                    m_TargetFileCheckOptions = new Dictionary<string, CheckOptions>(ParseTargetExtensionList(File.ReadAllText(targetExtensionListFile)));
                }
                catch (Exception e)
                {
                    Console.Error.WriteLine(string.Format("Error: Failed to read extensions list file '{0}'", targetExtensionListFile));
                    Console.Error.WriteLine(e.Message);
                    throw e;
                }
            }
        }

        // 整形対象に応じて適切なチェックオプションを返します。
        public CheckOptions GetFileCheckOptions(string filename)
        {
            // 整形対象が無指定ならば、すべてのファイルについて、全チェックを適用
            if (m_TargetFileCheckOptions == null)
            {
                return CheckOptions.All;
            }

            // 拡張子に応じたチェックオプションを返す
            string extension = Path.GetExtension(filename).ToLower();
            return m_TargetFileCheckOptions.ContainsKey(extension) ? m_TargetFileCheckOptions[extension] : CheckOptions.None;
        }

        // コード整形の対象となるファイルの拡張子を列挙したテキストをパースし、拡張子とチェックオプションの組を返します。
        private static IDictionary<string, CheckOptions> ParseTargetExtensionList(string targetExtensionList)
        {
            var ret = new Dictionary<string, CheckOptions>();

            // 入力形式は行ごとに "<拡張子>\t<チェックオプション文字列>" となっている
            // まずは行とタブで区切る
            var entries = targetExtensionList
                .Split(new[] { "\r\n", "\n" }, StringSplitOptions.RemoveEmptyEntries)
                .Select(line => line.Split(new[] { '\t' }, StringSplitOptions.RemoveEmptyEntries));

            foreach (var entry in entries)
            {
                var extension = entry[0].ToLower();
                var checkOptions = CheckOptions.None;

                if (entry.Length > 1)
                {
                    // チェックオプション文字列が存在すれば、それを CheckOptions 列挙体の値に変換
                    var optionStrings = entry[1].Split(new[] { '|' }, StringSplitOptions.RemoveEmptyEntries).Select(x => x.Trim());

                    foreach (var optionString in optionStrings)
                    {
                        CheckOptions option;
                        if (Enum.TryParse(optionString, out option))
                        {
                            checkOptions |= option;
                        }
                        else
                        {
                            throw new ArgumentException(string.Format("Unknown check option: {0}", optionString));
                        }
                    }
                }
                else
                {
                    // チェックオプション文字列がなければ全項目をチェック
                    checkOptions = CheckOptions.All;
                }

                ret[extension] = checkOptions;
            }

            return ret;
        }
    }
}
