﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace CodingCheckerUtil
{
    public static class FormatterUtil
    {
        private const int IndentSize = 4;   //インデントタブのスペース化時のスペース数

        /// <summary>
        /// バイト列のエンコーディングが UTF8 BOM 付きであるかどうかを判定します。
        /// UTF8 BOM 付きである場合に true を返します。
        /// </summary>
        /// <param name="content"></param>
        /// <returns></returns>
        public static bool CheckBOM(byte[] content, bool outputError = true, string errorHeader = "")
        {
            bool ret;
            if (content.Length >= 3)
            {
                // bom must be 0xEF 0xBB 0xBF
                ret = (content[0] == 0xEF && content[1] == 0xBB && content[2] == 0xBF);
            }
            else
            {
                ret = false;
            }

            if (outputError && !ret)
            {
                Console.Error.WriteLine(errorHeader + "Error: Encoding is wrong. Encoding must be UTF-8(BOM)");
            }

            return ret;
        }

        /// <summary>
        /// 改行コードが CRLF になっているかチェックし、修正する必要があれば true を返します。
        /// </summary>
        /// <param name="content"></param>
        /// <returns></returns>
        public static bool CheckLineBreak(string content)
        {
            string wrongCRPattern = @"\r(?!\n)";  // CR 改行単独パターン
            string wrongLFPattern = @"(?<!\r)\n"; // LF 改行単独パターン

            bool lineBreakReplaceFlag = Regex.IsMatch(content, wrongCRPattern, RegexOptions.Singleline);
            lineBreakReplaceFlag = lineBreakReplaceFlag | Regex.IsMatch(content, wrongLFPattern, RegexOptions.Singleline);

            // 後続の ConvertLfToCrlf で変換される
            return lineBreakReplaceFlag;
        }

        /// <summary>
        /// 改行コードを全て LineFeed に変換します。
        /// </summary>
        /// <param name="content"></param>
        public static void ConvertLineBreakToLf(ref string content)
        {
            // 改行コードの修正
            string returnPattern = @"\r\n|\r|\n"; // 改行コードパターン
            content = Regex.Replace(
                content,
                returnPattern,
                "\n", // $ が正しく使えるようにいったん\nに改行コードを変換
                RegexOptions.Singleline); //\nも認識できるように
        }

        /// <summary>
        /// LineFeed を全て CRLF に変換します。
        /// </summary>
        /// <param name="content"></param>
        public static void ConvertLfToCrlf(ref string content)
        {
            //改行コードの修正
            string returnPattern = @"\r\n|\r|\n"; // 改行コードパターン
            content = Regex.Replace(
                content,
                returnPattern,
                "\r\n", // 正しい改行コードへ
                RegexOptions.Singleline); //\nも認識できるように
        }

        /// <summary>
        /// 行頭タブをスペースに修正します。
        /// 修正が入った場合、true を返します。
        /// </summary>
        /// <param name="content"></param>
        /// <returns></returns>
        public static bool FormatTabToSpace(ref string content)
        {
            bool tabReplacedFlag = false;
            string headTabPattern = @"^(\t+)"; //行の先頭にタブがあるパターン
            content = Regex.Replace(
                content,
                headTabPattern,
                (Match m) =>
                {
                    tabReplacedFlag = true;
                    const int groupNumber = 0;
                    string key = m.Groups[groupNumber].ToString();
                    string indentSpace = string.Empty;
                    indentSpace = indentSpace.PadRight(IndentSize * key.Length); //タブの個数に対応したインデント用スペースを作成する。
                    return indentSpace;
                },
                RegexOptions.Multiline);

            return tabReplacedFlag;
        }

        /// <summary>
        /// 行末スペースを削除します。
        /// 修正が入った場合、true を返します。
        /// </summary>
        /// <param name="content"></param>
        /// <returns></returns>
        public static bool FormatFootSpace(ref string content)
        {
            bool footSpaceReplacedFlag = false;
            string footSpacePattern = @"[ \t]+$"; //行の最後に空白があるパターン
            content = Regex.Replace(
                content,
                footSpacePattern,
                (Match m) =>
                {
                    footSpaceReplacedFlag = true;
                    return string.Empty; // 空白の削除をする
                },
                RegexOptions.Multiline);

            return footSpaceReplacedFlag;
        }

        /// <summary>
        /// ファイル末尾に改行がない場合、改行を挿入します。
        /// 修正が入った場合、true を返します。
        /// </summary>
        /// <param name="content"></param>
        /// <returns></returns>
        public static bool FormatFootLineBreak(ref string content)
        {
            bool insertedReternEndFlag = false;
            string notReternEndPattern = @"\n\z";
            if (!Regex.IsMatch(
                content,
                notReternEndPattern,
                RegexOptions.Singleline | RegexOptions.RightToLeft)) //末尾チェックのため後ろから検索をかけます。
            {
                insertedReternEndFlag = true;
                content += "\n";
            }
            return insertedReternEndFlag;
        }
    }
}
