﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VsSolutionLibrary
{
    /// <summary>
    /// ソリューションファイルのパーサです。
    ///
    /// 等号(=) / コンマ(,) / ダブルクォーテーション(") などの特殊な意味を持つ文字が
    /// パス名などの文字列に含まれるケースには対応していません。
    /// </summary>
    internal class Parser
    {
        private static readonly string Header = "Microsoft Visual Studio Solution File, Format Version 12.00";

        /// <summary>
        /// ソリューションファイルを構成する文を表すクラス。
        /// ここで、ソリューションファイルの1つの行が1つの文を表すとしている。
        /// </summary>
        private class Statement
        {
            public enum StatementType
            {
                ProjectSection,
                EndProjectSection,
                Project,
                EndProject,
                GlobalSection,
                EndGlobalSection,
                Global,
                EndGlobal,
                Default
            }

            public StatementType Type { get; private set; }

            /// <summary>
            /// Statement を構成する文字列において、= の左側の部分
            /// </summary>
            public string Lhs { get; private set; }

            /// <summary>
            /// Statement を構成する文字列において、= の右側の部分
            /// </summary>
            public string Rhs { get; private set; }

            private string GetLhs(string text)
            {
                int index = text.IndexOf('=');

                if (index < 0)
                {
                    throw new ArgumentException("文に = が含まれていません。");
                }
                else if (index == 0)
                {
                    throw new ArgumentException("文の左辺が存在しません。");
                }
                else
                {
                    return text.Remove(index - 1).Trim();
                }
            }

            private string GetRhs(string text)
            {
                int index = text.IndexOf('=');

                if (index < 0)
                {
                    throw new ArgumentException("文に = が含まれていません。");
                }
                else if (index == text.Length - 1)
                {
                    throw new ArgumentException("文の右辺が存在しません。");
                }
                else
                {
                    return text.Substring(index + 1).Trim();
                }
            }

            public Statement(string text)
            {
                if (text.StartsWith("ProjectSection"))
                {
                    Type = StatementType.ProjectSection;
                    Lhs = GetLhs(text);
                    Rhs = GetRhs(text);
                }
                else if (text.StartsWith("EndProjectSection"))
                {
                    Type = StatementType.EndProjectSection;
                }
                else if (text.StartsWith("Project"))
                {
                    Type = StatementType.Project;
                    Lhs = GetLhs(text);
                    Rhs = GetRhs(text);
                }
                else if (text.StartsWith("EndProject"))
                {
                    Type = StatementType.EndProject;
                }
                else if (text.StartsWith("GlobalSection"))
                {
                    Type = StatementType.GlobalSection;
                    Lhs = GetLhs(text);
                    Rhs = GetRhs(text);
                }
                else if (text.StartsWith("EndGlobalSection"))
                {
                    Type = StatementType.EndGlobalSection;
                }
                else if (text.StartsWith("Global"))
                {
                    Type = StatementType.Global;
                }
                else if (text.StartsWith("EndGlobal"))
                {
                    Type = StatementType.EndGlobal;
                }
                else
                {
                    Type = StatementType.Default;
                    Lhs = GetLhs(text);
                    Rhs = GetRhs(text);
                }
            }
        }

        private class SyntaxTree
        {
            /// <summary>
            /// Project を表す SyntaxTree の要素
            /// </summary>
            public class ProjectElement
            {
                public Statement ProjectStatement { get; private set; }
                public List<ProjectSectionElement> ProjectSections { get; set; }

                /// <summary>
                /// キューから Statement を取り出してパースし、ProjectElement を作成する。
                /// </summary>
                /// <param name="statementQueue">先頭に Project 文を含む Statement のキュー</param>
                public ProjectElement(Queue<Statement> statementQueue)
                {
                    ProjectSections = new List<ProjectSectionElement>();

                    Statement statement = statementQueue.Dequeue();
                    Debug.Assert(
                        statement.Type == Statement.StatementType.Project,
                        "Project 文が予期された場所に他の文があります。");

                    ProjectStatement = statement;

                    while (statementQueue.Count > 0)
                    {
                        statement = statementQueue.Peek();

                        if (statement.Type == Statement.StatementType.EndProject)
                        {
                            statementQueue.Dequeue();
                            return;
                        }
                        else if (statement.Type == Statement.StatementType.ProjectSection)
                        {
                            ProjectSections.Add(new ProjectSectionElement(statementQueue));
                        }
                        else
                        {
                            throw new ArgumentException("ソリューションの Project 要素に不正な文が含まれています。");
                        }
                    }

                    throw new ArgumentException("ソリューションの Project 文に対応する EndProject 文がありません。");
                }
            }

            /// <summary>
            /// ProjectSection を表す SyntaxTree の要素
            /// </summary>
            public class ProjectSectionElement
            {
                public enum ProjectSectionType
                {
                    ProjectDependencies
                }

                public ProjectSectionType Type { get; private set; }

                public List<Statement> Statements { get; private set; }

                public ProjectSectionElement(Queue<Statement> statementQueue)
                {
                    Statements = new List<Statement>();

                    Statement statement = statementQueue.Dequeue();
                    Debug.Assert(
                        statement.Type == Statement.StatementType.ProjectSection,
                        "ProjectSection 文が予期された場所に他の文があります。");

                    if (Utility.GetLhsArgument(statement) != "ProjectDependencies")
                    {
                        string s = Utility.GetLhsArgument(statement);
                        throw new ArgumentException("ソリューションの Project 文の左辺引数が不正です。");
                    }
                    Type = ProjectSectionType.ProjectDependencies;

                    while (statementQueue.Count > 0)
                    {
                        statement = statementQueue.Peek();

                        if (statement.Type == Statement.StatementType.EndProjectSection)
                        {
                            statementQueue.Dequeue();
                            return;
                        }
                        else if (statement.Type == Statement.StatementType.Default)
                        {
                            Statements.Add(statement);
                            statementQueue.Dequeue();
                        }
                        else
                        {
                            throw new ArgumentException("ソリューションの ProjectSection 要素に不正な文が含まれています。");
                        }
                    }

                    throw new ArgumentException("ソリューションの ProjectSection 文に対応する EndProjectSection 文がありません。");
                }
            }

            /// <summary>
            /// Global を表す SyntaxTree の要素
            /// </summary>
            public class GlobalElement
            {
                public List<GlobalSectionElement> PreSolutionGlobalSections { get; set; }
                public List<GlobalSectionElement> PostSolutionGlobalSections { get; set; }

                /// <summary>
                /// キューから Statement を取り出してパースし、GlobalElement を作成する。
                /// </summary>
                /// <param name="statementQueue">先頭に Global 文を含む Statement のキュー</param>
                public GlobalElement(Queue<Statement> statementQueue)
                {
                    PreSolutionGlobalSections = new List<GlobalSectionElement>();
                    PostSolutionGlobalSections = new List<GlobalSectionElement>();

                    Statement statement = statementQueue.Dequeue();
                    Debug.Assert(
                        statement.Type == Statement.StatementType.Global,
                        "Global 文が予期された場所に他の文があります。");

                    while (statementQueue.Count > 0)
                    {
                        statement = statementQueue.Peek();

                        if (statement.Type == Statement.StatementType.EndGlobal)
                        {
                            statementQueue.Dequeue();
                            return;
                        }
                        else if (statement.Type == Statement.StatementType.GlobalSection)
                        {
                            if (statement.Rhs == "preSolution")
                            {
                                PreSolutionGlobalSections.Add(new GlobalSectionElement(statementQueue));
                            }
                            else if (statement.Rhs == "postSolution")
                            {
                                PostSolutionGlobalSections.Add(new GlobalSectionElement(statementQueue));
                            }
                            else
                            {
                                throw new ArgumentException("ソリューションの GlobalSection 文が不正です。");
                            }
                        }
                        else
                        {
                            throw new ArgumentException("ソリューションの Global 要素に不正な文が含まれています。");
                        }
                    }

                    throw new ArgumentException("ソリューションの Global 文に対応する EndGlobal 文がありません。");
                }
            }

            /// <summary>
            /// GlobalSection を表す SyntaxTree の要素
            /// </summary>
            public class GlobalSectionElement
            {
                public enum GlobalSectionType
                {
                    SolutionConfigurationPlatforms,
                    ProjectConfigurationPlatforms,
                    SolutionProperties,
                    NestedProjects
                }

                public GlobalSectionType Type { get; private set; }

                public List<Statement> Statements { get; private set; }

                public GlobalSectionElement(Queue<Statement> statementQueue)
                {
                    Statements = new List<Statement>();

                    Statement statement = statementQueue.Dequeue();
                    Debug.Assert(
                        statement.Type == Statement.StatementType.GlobalSection,
                        "GlobalSection 文が予期された場所に他の文があります。");

                    if (Utility.GetLhsArgument(statement) == "ProjectConfigurationPlatforms")
                    {
                        Type = GlobalSectionType.ProjectConfigurationPlatforms;
                    }
                    else if (Utility.GetLhsArgument(statement) == "SolutionConfigurationPlatforms")
                    {
                        Type = GlobalSectionType.SolutionConfigurationPlatforms;
                    }
                    else if (Utility.GetLhsArgument(statement) == "SolutionProperties")
                    {
                        Type = GlobalSectionType.SolutionProperties;
                    }
                    else if (Utility.GetLhsArgument(statement) == "NestedProjects")
                    {
                        Type = GlobalSectionType.NestedProjects;
                    }
                    else
                    {
                        throw new ArgumentException("ソリューションの Project 文の左辺引数が不正です。");
                    }

                    while (statementQueue.Count > 0)
                    {
                        statement = statementQueue.Peek();

                        if (statement.Type == Statement.StatementType.EndGlobalSection)
                        {
                            statementQueue.Dequeue();
                            return;
                        }
                        else if (statement.Type == Statement.StatementType.Default)
                        {
                            Statements.Add(statement);
                            statementQueue.Dequeue();
                        }
                        else
                        {
                            throw new ArgumentException("ソリューションの GlobalSection 要素に不正な文が含まれています。");
                        }
                    }

                    throw new ArgumentException("ソリューションの GlobalSection 文に対応する EndGlobalSection 文がありません。");
                }
            }

            /// <summary>
            /// Statement のリストから 構文木を生成する
            /// </summary>
            /// <param name="statements">構文木を記述した Statement のリスト</param>
            public SyntaxTree(IEnumerable<Statement> statements)
            {
                Projects = new List<ProjectElement>();
                RootStatements = new List<Statement>();

                // パース用のキューを作成
                var statementQueue = new Queue<Statement>(statements);

                // Statement を順番に処理して、構文木を作成
                while (statementQueue.Count > 0)
                {
                    Statement statement = statementQueue.Peek();

                    if (statement.Type == Statement.StatementType.Project)
                    {
                        Projects.Add(new ProjectElement(statementQueue));
                    }
                    else if (statement.Type == Statement.StatementType.Global)
                    {
                        if (Global != null)
                        {
                            throw new ArgumentException("ソリューションに複数の Global 文が含まれています。");
                        }
                        Global = new GlobalElement(statementQueue);
                    }
                    else if (statement.Type == Statement.StatementType.Default)
                    {
                        RootStatements.Add(statement);
                        statementQueue.Dequeue();
                    }
                    else
                    {
                        throw new ArgumentException("ソリューションのルートに不正な文が含まれています。");
                    }
                }
            }

            public List<ProjectElement> Projects { get; private set; }

            public GlobalElement Global { get; private set; }

            // バージョン情報などの、Project / Global に所属しない文
            public List<Statement> RootStatements { get; private set; }
        }

        public static void Parse(string text, VsSolution solution)
        {
            IEnumerable<Statement> statements = CreateStatementListFromText(text);
            SyntaxTree syntaxTree = new SyntaxTree(statements);

            // 規定の順番で構文を処理する
            ProcessRootStatement(solution, syntaxTree.RootStatements);

            if (syntaxTree.Global != null && syntaxTree.Global.PreSolutionGlobalSections != null)
            {
                foreach (var globalSection in syntaxTree.Global.PreSolutionGlobalSections)
                {
                    ProcessGlobalSection(solution, globalSection);
                }
            }

            foreach (var project in syntaxTree.Projects)
            {
                ProcessProject(solution, project);
            }

            if (syntaxTree.Global != null && syntaxTree.Global.PostSolutionGlobalSections != null)
            {
                foreach (var globalSection in syntaxTree.Global.PostSolutionGlobalSections)
                {
                    ProcessGlobalSection(solution, globalSection);
                }
            }
        }

        private static IEnumerable<Statement> CreateStatementListFromText(string text)
        {
            var statements = new List<Statement>();
            var reader = new StringReader(text);
            bool headerFound = false;

            // 1行ずつ処理する
            string line;
            while ((line = reader.ReadLine()) != null)
            {
                // 先頭と末尾の空白を削除
                line = line.Trim();

                // 空行は無視
                if (line.Length == 0)
                {
                    continue;
                }

                // # から始まる行はコメント行なので無視
                if (line[0] == '#')
                {
                    continue;
                }

                // ヘッダ行の処理
                if (!headerFound)
                {
                    if (line == Header)
                    {
                        headerFound = true;
                        continue;
                    }
                    else
                    {
                        throw new ArgumentException("ソリューションファイルのヘッダが不正です。");
                    }
                }

                // 行の種類を判別し、Statement オブジェクトを作成
                Statement statement = new Statement(line);
                statements.Add(statement);
            }

            return statements;
        }

        /// <summary>
        /// SyntaxTree の Project / Global 以外の情報に従って、VsSolutionオブジェクトを更新する。
        /// </summary>
        /// <param name="solution">操作対象の VsSolution オブジェクト</param>
        /// <param name="statements">SyntaxTree の Project / Global 以外の情報</param>
        private static void ProcessRootStatement(VsSolution solution, IEnumerable<Statement> statements)
        {
            foreach (var statement in statements)
            {
                if (statement.Lhs == "VisualStudioVersion")
                {
                    if (statement.Rhs.StartsWith("12."))
                    {
                        solution.Version = VsSolutionVersion.VisualStudio2013;
                    }
                    else if (statement.Rhs.StartsWith("14."))
                    {
                        solution.Version = VsSolutionVersion.VisualStudio2015;
                    }
                    else if (statement.Rhs.StartsWith("15."))
                    {
                        solution.Version = VsSolutionVersion.VisualStudio2017;
                    }
                    else
                    {
                        throw new ArgumentException("ソリューションの VisualStudioVersion に未対応の値が設定されています。。");
                    }
                }
                else if (statement.Lhs == "MinimumVisualStudioVersion")
                {
                }
                else
                {
                    throw new ArgumentException("ソリューションのルートに不正な文が含まれています。");
                }
            }
        }

        /// <summary>
        /// SyntaxTree のプロジェクト情報に従って、VsSolution オブジェクトにプロジェクトを追加する。
        /// </summary>
        /// <param name="solution">操作対象の VsSolution オブジェクト</param>
        /// <param name="project">SyntaxTree のプロジェクト情報</param>
        private static void ProcessProject(VsSolution solution, SyntaxTree.ProjectElement project)
        {
            // Project 文の左辺
            Guid projectTypeGuid = Utility.GetLhsGuidArgument(project.ProjectStatement);
            VsProjectType projectType = VsProjectTypeUtility.GetProjectTypeFromGuid(projectTypeGuid);

            // Project 文の右辺
            string[] rhsStrings = Utility.GetRhsStrings(project.ProjectStatement);
            string projectName = rhsStrings[0];
            string projectPath = rhsStrings[1];
            Guid projectGuid = new Guid(rhsStrings[2].Trim(new char[] { '{', '}' }));

            // Dependency
            List<Guid> dependencies = new List<Guid>();
            foreach (var section in project.ProjectSections)
            {
                Debug.Assert(
                    section.Type == SyntaxTree.ProjectSectionElement.ProjectSectionType.ProjectDependencies,
                    "ProjectDependencies セクションが予期された場所に他のセクションがあります。");

                foreach (var statement in section.Statements)
                {
                    if (statement.Lhs != statement.Rhs)
                    {
                        throw new ArgumentException("ソリューションの ProjectSection に不正な要素が含まれています。");
                    }

                    dependencies.Add(new Guid(statement.Lhs));
                }
            }

            // VsSolution にプロジェクトを追加
            solution.AddProject(projectGuid, new VsSolutionProjectProperty(projectType, projectName, projectPath, dependencies));
        }

        /// <summary>
        /// SyntaxTree の GlobalSection 情報に従って、VsSolution オブジェクトを更新する。
        /// </summary>
        /// <param name="solution">操作対象の VsSolution オブジェクト</param>
        /// <param name="globalSection">SyntaxTree の GlobalSection 情報</param>
        private static void ProcessGlobalSection(VsSolution solution, SyntaxTree.GlobalSectionElement globalSection)
        {
            switch (globalSection.Type)
            {
                case SyntaxTree.GlobalSectionElement.GlobalSectionType.ProjectConfigurationPlatforms:
                    // プロジェクト・コンフィギュレーションの追加
                    foreach (var statement in globalSection.Statements)
                    {
                        // 以下のどちらかの形式になっている
                        // {ProjectGuid}.SolutionConfig|SolutionPlatform.ActiveCfg = ProjectConfig|ProjectPlatform
                        // {ProjectGuid}.SolutionConfig|SolutionPlatform.Build.0 = ProjectConfig|ProjectPlatform

                        string[] lhsStrings = statement.Lhs.Split('.');
                        if (lhsStrings.Length < 2)
                        {
                            throw new ArgumentException("ソリューションに不正な ProjectConfiguration が含まれています。");
                        }

                        Guid projectGuid = new Guid(lhsStrings[0]);
                        VsConfigurationPair solutionConfiguration = new VsConfigurationPair(lhsStrings[1]);

                        VsProjectConfiguration oldProjectConfiguration = solution.GetProjectConfiguration(solutionConfiguration, projectGuid);
                        if (lhsStrings.Length == 3 && lhsStrings[2] == "ActiveCfg")
                        {
                            solution.SetProjectConfiguration(
                                solutionConfiguration,
                                projectGuid,
                                new VsProjectConfiguration(new VsConfigurationPair(statement.Rhs), oldProjectConfiguration.EnableBuild));
                        }
                        else if (lhsStrings.Length == 4 && lhsStrings[2] == "Build" && lhsStrings[3] == "0")
                        {
                            solution.SetProjectConfiguration(
                                solutionConfiguration,
                                projectGuid,
                                new VsProjectConfiguration(oldProjectConfiguration.ConfigurationPair, true));
                        }
                        else
                        {
                            throw new ArgumentException("ソリューションに不正な ProjectConfiguration が含まれています。");
                        }
                    }
                    break;

                case SyntaxTree.GlobalSectionElement.GlobalSectionType.SolutionConfigurationPlatforms:
                    // ソリューション・コンフィギュレーションの追加
                    foreach (var statement in globalSection.Statements)
                    {
                        if (statement.Lhs != statement.Rhs)
                        {
                            throw new ArgumentException("ソリューションに不正な SolutionConfiguration が含まれています。");
                        }

                        string[] strings = statement.Lhs.Split('|');
                        var solutionConfiguration = new VsConfigurationPair(strings[0], strings[1]);
                        solution.AddSolutionConfiguration(solutionConfiguration);
                    }
                    break;

                case SyntaxTree.GlobalSectionElement.GlobalSectionType.SolutionProperties:
                    // ソリューション・プロパティの追加
                    foreach (var statement in globalSection.Statements)
                    {
                        solution.SolutionProperties.Add(statement.Lhs, statement.Rhs);
                    }
                    break;
                case SyntaxTree.GlobalSectionElement.GlobalSectionType.NestedProjects:
                    // ソリューション・プロパティの追加
                    foreach (var statement in globalSection.Statements)
                    {
                        Guid parent = new Guid(statement.Rhs);
                        Guid child = new Guid(statement.Lhs);
                        VsNestRelation relation = new VsNestRelation(parent, child);
                        solution.AddNestRelation(relation);
                    }
                    break;

                default:
                    throw new ArgumentException();
            }
        }

        /// <summary>
        /// ユーティリティクラス
        /// </summary>
        private static class Utility
        {
            // Statement の左辺の引数 (括弧に囲まれた文字列) を返す
            public static string GetLhsArgument(Statement statement)
            {
                int start = statement.Lhs.IndexOf('(');
                int end = statement.Lhs.IndexOf(')');

                if (!(0 < start && start < end && end == statement.Lhs.Length - 1))
                {
                    throw new ArgumentException("ソリューションに不正な Argument が含まれています。");
                }

                return statement.Lhs.Substring(start + 1, end - (start + 1));
            }

            // Statement の左辺の引数 (括弧に囲まれた文字列) を GUID にして返す
            public static Guid GetLhsGuidArgument(Statement statement)
            {
                string s = GetLhsArgument(statement);

                if (!(s.Length > 2 && s[0] == '"' && s[s.Length - 1] == '"'))
                {
                    throw new ArgumentException("ソリューションに不正な GuidArgument が含まれています。");
                }

                return new Guid(s.Substring(1, s.Length - 2));
            }

            // Statement の右辺の引数をパースして、文字列の配列を返す
            public static string[] GetRhsStrings(Statement statement)
            {
                string[] strings = statement.Rhs.Split(',');

                // 配列の各要素を整形
                for (int i = 0; i < strings.Length; i++)
                {
                    string s = strings[i];

                    // 前後の空白を取り除く
                    s = s.Trim();

                    // 前後のダブルクォーテーションを取り除く
                    if (!(s.Length > 2 && s[0] == '"' && s[s.Length - 1] == '"'))
                    {
                        throw new ArgumentException("ソリューションに不正な文字列リストが含まれています。");
                    }
                    strings[i] = s.Substring(1, s.Length - 2);
                }

                return strings;
            }
        }
    }
}
