<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Verifies the test game *.nsp on the device was loaded correctly, by looking at the device logs

    .DESCRIPTION
        - Copies the device logs from the device, and looks for a string to validate the *.nsp was loaded correctly
        
#>

param
(
    [string]
    # SDK root directory
    $sigloRoot =
    ($env:NINTENDO_SDK_ROOT | ? { $_ } | % { $_.Trim() }),

    [string]
    # Device's IP address
    $TargetAddress =
    ($env:TARGET_ADDRESS | ? { $_ } | % { $_.Trim() }),

    [string]
    # string to look for in device logs to verify the test game was run correctly
    #   eg. "[Race(4) Scene] After sceneEnter"
    $deviceLogString =
    ($env:DEVICE_LOG_STRING)    
)

# global values
$NX_AGING_WORKING_DIR = "$sigloRoot\Integrate\Outputs\AnyCPU\Tools\NxAgingTools\NxAgingHelperCommand\Release"

# reset the device in case there was an exception during the test run
& Tools\CommandLineTools\ControlTarget.exe reset --hard --target $TargetAddress
& Tools\CommandLineTools\ControlTarget.exe connect

# disable logging on the device
Write-Host "Disable logging on the device SD card"
& Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- debug disable-sd-card-logging
& Tools\CommandLineTools\ControlTarget.exe reset --hard --target $TargetAddress
& Tools\CommandLineTools\ControlTarget.exe connect

# give the SD card time to be ready for reading
Start-sleep -s 1

# copy the device logs from the SD card to the host
Write-Host "--- Copying logs for sleep/wake"
Start-Process -FilePath "Tools\CommandLineTools\RunOnTarget.exe" -ArgumentList "./Tests/Outputs/NX-NXFP2-a64/Tests/GetSdCardFiles/Develop/GetSdCardFiles.nsp -- --read_dir=sd:/NxBinLogs --write_dir=$NX_AGING_WORKING_DIR" -Wait
Start-Process -FilePath "Tests\Outputs\AnyCPU\Tools\ParseNxBinLogs\Release\ParseNxBinLogs.exe" -ArgumentList "$NX_AGING_WORKING_DIR" -Wait

# look for a specific string in the device logs to verify the game was loaded correctly
$stringToFind = $null
$flagStringFound = 0
$logFileNameArray = Get-ChildItem -Path $NX_AGING_WORKING_DIR -Filter *.nxbinlog.log -Recurse
foreach ($logFileName in $logFileNameArray)
{
    $stringToFind = Select-String -Path $NX_AGING_WORKING_DIR\$logFileName -Pattern $deviceLogString -Encoding ascii -SimpleMatch
    if ($stringToFind)
    {
        # the search string was found
        $flagStringFound = 1
        break
    }
}

if ($flagStringFound -eq 0)
{
    Write-Host "--- The test game was not loaded correctly"
    exit 1
}

