﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

param (
    [string]$RyndaVersion = "",
    [string]$GccVersion = "",
    [string]$MsysVersion = "",
    [string]$DtcVersion = "",
    [string]$NactVersion = ""
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${ScriptDirectoryPath}\Modules\Error"
Import-Module "${ScriptDirectoryPath}\Modules\Path"
Import-Module "${ScriptDirectoryPath}\Modules\BuildSettings"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

function ExtractZip {
    param (
        [parameter(Mandatory=$True,Position=1)] [String] $ZipFile,
        [parameter(Mandatory=$True,Position=2)] [String] $ExtractDestination
    )

    $SevenZip = Get-7ZipPath

    try
    {
        Invoke-Expression "$SevenZip x $ZipFile -o$ExtractDestination"
    }
    finally
    {
        if ($? -eq $False -or $LastExitCode -ne 0)
        {
            $ExtractSubDirectory = $ZipFile.Replace((Get-Item $ZipFile).Extension,"")
            if ((Test-Path -Path $ExtractSubDirectory)){
                Remove-Item $ExtractSubDirectory -Force -Recurse
            }
            Remove-Item $ZipFile -Force
            throw "Could not extract $ZipFile ($LastExitCode)!"
        }
    }
}

function GetArchivePath {
    param (
        [parameter(Mandatory=$True,Position=1)] [String] $RelativePath
    )

    if ($Env:SIGLO_TOOLCHAINS_SOURCE){
        $SetupRootCustom = $Env:SIGLO_TOOLCHAINS_SOURCE
        if (Test-Path -Path "$SetupRootCustom\$RelativePath"){
            return "$SetupRootCustom\$RelativePath"
        }
    }

    $EmailAddress       = &(Get-GitPath) config user.email
    if($EmailAddress -ne $null)
    {
        $EmailDomain        = $EmailAddress.Split("{@}")[1]

        switch ($EmailDomain){
            ntd.nintendo.com {
                $SetupRootAllCDN       = "\\noa01dept01.noa.nintendo.com\ntd_dept\projects\n_proj\Siglo\ToolChains\ForAll\setup"
                $SetupRootNXUsersCDN   = "\\noa01dept01.noa.nintendo.com\ntd_dept\projects\n_proj\Siglo\ToolChains\ForNXUsers\setup"
            }
        }
    }

    $SetupRootAll       = "\\devdatasrv2.ncl.nintendo.co.jp\share-siglo$\ForAll\setup"
    $SetupRootNXUsers   = "\\devdatasrv2.ncl.nintendo.co.jp\share-siglo$\ForNXUsers\setup"

    if (($SetupRootAllCDN) -and (Test-Path -Path "$SetupRootAllCDN\$RelativePath")){
        return "$SetupRootAllCDN\$RelativePath"
    }
    if (($SetupRootNXUsersCDN) -and (Test-Path -Path "$SetupRootNXUsersCDN\$RelativePath")){
        return "$SetupRootNXUsersCDN\$RelativePath"
    }
    if (Test-Path -Path "$SetupRootAll\$RelativePath"){
        return "$SetupRootAll\$RelativePath"
    }
    if (Test-Path -Path "$SetupRootNXUsers\$RelativePath"){
        return "$SetupRootNXUsers\$RelativePath"
    }

    Write-Host Could not find a required archive $RelativePath in the Siglo shared folder.

    exit 1
}

if ($RyndaVersion -eq "")
{
    $RyndaVersion = Get-BuildToolChainVersion("Rynda")
}

if ($GccVersion -eq "")
{
    $GccVersion = Get-BuildToolChainVersion("SigloGcc")
}

if ($MsysVersion -eq "")
{
    $MsysVersion = Get-BuildToolChainVersion("SigloMsys")
}

if ($DtcVersion -eq "")
{
    $DtcVersion = Get-BuildToolChainVersion("SigloDtc")
}

if ($NactVersion -eq "")
{
    $NactVersion = Get-NactVersion
}

Write-Host Required toolchains: RYNDA="$RyndaVersion" GCC="$GccVersion" MSYS="$MsysVersion" DTC="$DtcVersion" Nact="$NactVersion"

# ToolChains のパスを解決
$ToolChainsRoot = "$(Get-NintendoSdkRootPath)\ToolChains"
if ($Env:SIGLO_TOOLCHAINS_ROOT){
    $ToolChainsRoot = $Env:SIGLO_TOOLCHAINS_ROOT
}

# ToolChains ディレクトリがなければ作成
if (!(Test-Path -Path "$ToolChainsRoot")){
    mkdir "${ToolChainsRoot}"
}

if (!(Test-Path -Path "$ToolChainsRoot\$GccVersion")){
    Write-Host SigloGCC "$GccVersion" is required

    $source =  GetArchivePath("compiler\old\$GccVersion.zip")
    if (!(Test-Path -Path "$ToolChainsRoot\$GccVersion.zip")){
        Write-Host "Copying $source, please wait..."
        robocopy (Split-Path $source) $ToolChainsRoot (Split-Path -Leaf $source) /ETA /NJH /NJS /NDL /R:100
    }
    ExtractZip ("$ToolChainsRoot\$GccVersion.zip") $ToolChainsRoot
}

if (!(Test-Path -Path "$ToolChainsRoot\$RyndaVersion")){
    Write-Host RYNDA "$RyndaVersion" is required

    $source =  GetArchivePath("compiler\Rynda\$RyndaVersion.zip")
    if (!(Test-Path -Path "$ToolChainsRoot\$RyndaVersion.zip")){
        Write-Host "Copying $source, please wait..."
        robocopy (Split-Path $source) $ToolChainsRoot (Split-Path -Leaf $source) /ETA /NJH /NJS /NDL /R:100
    }
    ExtractZip ("$ToolChainsRoot\$RyndaVersion.zip") $ToolChainsRoot
}

if (!(Test-Path -Path "$ToolChainsRoot\$MsysVersion")){
    Write-Host MSYS "$MsysVersion" is required

    $source =  GetArchivePath("msys\$MsysVersion.zip")
    if (!(Test-Path -Path "$ToolChainsRoot\$MsysVersion.zip")){
        Write-Host "Copying $source, please wait..."
        robocopy (Split-Path $source) $ToolChainsRoot (Split-Path -Leaf $source) /ETA /NJH /NJS /NDL /R:100
    }
    ExtractZip ("$ToolChainsRoot\$MsysVersion.zip") $ToolChainsRoot
}

if (!(Test-Path -Path "$ToolChainsRoot\$DtcVersion")){
    Write-Host SigloDtc "$DtcVersion" is required

    $source =  GetArchivePath("DeviceTreeCompiler\$DtcVersion.zip")
    if (!(Test-Path -Path "$ToolChainsRoot\$DtcVersion.zip")){
        Write-Host "Copying $source, please wait..."
        robocopy (Split-Path $source) $ToolChainsRoot (Split-Path -Leaf $source) /ETA /NJH /NJS /NDL /R:100
    }
    ExtractZip ("$ToolChainsRoot\$DtcVersion.zip") $ToolChainsRoot
}

if (!(Test-Path -Path "$ToolChainsRoot\flash-min-fp2")){
    Write-Host flash-min-fp2 is required

    $source =  GetArchivePath("SDEV\flash-min-fp2.zip")
    if (!(Test-Path -Path "$ToolChainsRoot\flash-min-fp2.zip")){
        Write-Host "Copying $source, please wait..."
        robocopy (Split-Path $source) $ToolChainsRoot (Split-Path -Leaf $source) /ETA /NJH /NJS /NDL /R:100
    }
    ExtractZip ("$ToolChainsRoot\flash-min-fp2.zip") $ToolChainsRoot
}

$nactRoot = "$(Get-NintendoSdkRootPath)\Build\Nact"
if (!(Test-Path -Path "$nactRoot\$NactVersion")){
    Write-Host Nact "$NactVersion" is required

    $source =  GetArchivePath("SigloNact\$NactVersion.zip")
    if (!(Test-Path -Path "$$ToolChainsRoot\$NactVersion.zip")){
        Write-Host "Copying $source, please wait..."
        robocopy (Split-Path $source) $ToolChainsRoot (Split-Path -Leaf $source) /ETA /NJH /NJS /NDL /R:100
    }
    ExtractZip ("$ToolChainsRoot\$NactVersion.zip") $nactRoot
}
