﻿<#
    Copyright (C)Nintendo. All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.
 
    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
    Bezelのアドオンドキュメント用の処理を行います。

    .DESCRIPTION
    Bezelのアドオンドキュメント用の処理を行います。
    
    ・同じファイル名のページを削除します。
    ・tocData.js ファイルの内容で、Bezelに関係する項目のみにします。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
[CmdletBinding()]
Param(
    # ドキュメントのルートディレクトリを指定します。
    [parameter(Mandatory = $true)]
    [string] $DocumentRoot,

    # Bezel のタイトルを指定します。
    [string] $Title = "NintendoWare Bezel Engine",

    # ログディレクトリを指定します。
    [string] $LogDir = "$DocumentRoot\Logs"
)

if (-not(Test-Path $LogDir))
{
    mkdir $LogDir -Force | Out-Null
}

$LogFile = "$LogDir\Select-BezelItem.log"

function Invoke-RewriteBezelEnginePageNotFoundLinks()
{
    param(
        [string]$PageRoot
    )
    $PageFiles = Get-ChildItem "$PageRoot\*.html"
    $PageLinkRegex = [Regex]"href=`"../Pages/(Page_\d+\.html)`""
    $PageLinkCallback = {
        param ($match)
        $PageFileName = $match.Groups[1].Value
        if (Test-Path $([IO.Path]::Combine($PageRoot, $PageFileName)))
        {
            # BezelEngine ドキュメント内にドキュメントファイルが存在する
            return $match.Value
        }
        # ファイルが存在しなかったので SDK 側ドキュメントへのリンクとみなす
        return $("href=`"{0}`"" -f "../../../Package/contents/Pages/$PageFileName")
    }

    Out-LogInfo "BezelEngine => NintendoSDK Documents Link Rewriter Start"
    foreach ($PageFile in $PageFiles)
    {
        $PageContent = [System.IO.File]::ReadAllText($PageFile.FullName)
        $PageFileName = $PageFile.Name
        $RewritedPageContent = $PageLinkRegex.Replace($PageContent, $PageLinkCallback)
        if ($PageContent -ne $RewritedPageContent)
        {
            [System.IO.File]::WriteAllText($PageFile.FullName, $RewritedPageContent, [System.Text.Encoding]::UTF8)
            Out-LogInfo "[ UPDATE ] $PageFileName"
        }
        else
        {
            Out-LogInfo "[NOCHANGE] $PageFileName"
        }
    }
    Out-LogInfo "BezelEngine => NintendoSDK Documents Link Rewriter Finished"
}

function Select-TableOfContentInSpecificPageTree()
{
    param(
        [string[]]$TocContent,
        [string]$RootPageTitle
    )
    $FilteredToc = New-Object System.Text.StringBuilder
    $InBezelDocument = $false
    $BezelTocLevel = 0
    $DescentPageLevel = {
        param($match)
        $g = $match.Groups
        $level = [int]$g[2].Value
        $level -= 1
        return @($g[1].Value, $level, $g[3].Value) -join ""
    }
    foreach ($Line in $TocContent)
    {
        if ($Line -match 'objTOC\.(add.*?)\([^,].*?,\s*(\d+),\s*"(.*?)",')
        {
            $AddMethod, $Level, $PageTitle = ($Matches[1], $Matches[2], $Matches[3])
            if ($AddMethod -eq "addExternalHeader")
            {
                $FilteredToc.AppendLine($Line) | Out-Null
                Out-LogInfo "[Toc] AddExternalHeader: $PageTitle"
                continue
            }
            if ($InBezelDocument -and ($Level -eq $BezelTocLevel))
            {
                $InBezelDocument = $false
            }
            if ($PageTitle -eq $RootPageTitle)
            {
                # BezelEngine ドキュメントの開始位置
                $BezelTocLevel = $Level
                $InBezelDocument = $true
            }
            if (!$InBezelDocument)
            {
                Out-LogInfo "[Toc] SkipPage: $PageTitle"
                continue
            }
            Out-LogInfo "[Toc] AddPage: $PageTitle"
            # 出力対象の add* メソッドの引数のうち、階層レベルをデクリメントする
            # (BezelEngine のルートページが Level=0 になるようにする)
            $Line = [Regex]::Replace($Line, '^(.*?\.add.*?\("ID_\d+",\s*)(\d+)(,.*?\);)', $DescentPageLevel)
        }
        $FilteredToc.AppendLine($Line) | Out-Null
    }
    
    # フィルタして取り除いたページのコメント部分 + 空白行を削除
    $FilteredTocContent = [Regex]::Replace(
        $FilteredToc.ToString(), 
        '//[^\r\n]*?\r\n\r\n', "",
        [System.Text.RegularExpressions.RegexOptions]::Singleline)
        
    return $FilteredTocContent
}

function Out-LogInfo()
{
    param([string]$Message)
    [IO.File]::AppendAllText($LogFile, "$Message`r`n", [Text.Encoding]::UTF8)
}

# BezelEngine ドキュメント内のページリンクについて、BezelEngine 内に存在しないものは SDK 側のドキュメントにあるとみなしてリンクを置き換えます。
Invoke-RewriteBezelEnginePageNotFoundLinks -PageRoot "$DocumentRoot\contents\Pages"

# tocData.js へのパス
$TocDataJsPath = Join-Path $DocumentRoot contents/tocData.js
$TocContentJs = Get-Content $TocDataJsPath
# tocData.js から $Title に指定されたページをルートとする目次のみを残して後はフィルタする
$FilteredTocContent = Select-TableOfContentInSpecificPageTree -TocContent $TocContentJs -RootPageTitle $Title
# フィルタ結果をファイルに書き出し
[System.IO.File]::WriteAllText($TocDataJsPath, $FilteredTocContent, [System.Text.Encoding]::UTF8)

Write-Host "Select-BezelItem finished, Log file location: $LogFile"