﻿<#
    Copyright (C)Nintendo. All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.
 
    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
    指定したラベルを持つページとその子孫を除いて、Confluence からエクスポートしたページと添付されたデータを削除します。

    .DESCRIPTION
    指定したラベルを持つページとその子孫を除いて、Confluence からエクスポートしたページと添付されたデータを削除します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>

[CmdletBinding()]
Param(
    # Confluence をエクスポートして取得した entities.xml へのパスを指定します。
    [parameter(Mandatory = $true)]
    [string] $XmlPath,

    # 削除せずに保持するラベルを指定します。
    # 指定したラベルを持つページと、その子孫が削除から除外されます。
    [parameter(Mandatory = $true)]
    [string] $FilterLabelName,

    # 削除対象のドキュメントのルートディレクトリを指定します。
    [parameter(Mandatory = $true)]
    [string] $TargetRoot,

    # 処理のログを出力するディレクリを指定します。
    [string] $LogDir,

    # $true ならファイルの削除処理を行いません。
    [switch] $DryRun
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Yaml"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

# あるページ ID に含まれる子孫ページの ID の一覧を連想配列に設定します。
function Build-DescendantsPageIds()
{
    param([string]$PageId, $PageIdSet, $PageIdTable)
    if ($PageIdSet.containsKey($PageId))
    {
        return
    }
    $PageIdSet.Add($PageId, $true)
    $pageInfo = $PageIdTable[$PageId]
    Out-LogInfo ("[AddDescendantsPage] {0} (ID=$PageId)" -f $pageInfo.Name)
    if ($pageInfo -eq $null)
    {
        return
    }
    foreach ($id in $pageInfo.ChildrenIds)
    {
        if ([string]::IsNullOrEmpty($id))
        {
            continue
        }
        Build-DescendantsPageIds -PageId $id -PageIdSet $PageIdSet -PageIdTable $PageIdTable
    }
}

# Confluence エクスポート XML から、ページ本体 ID => ページ本体の内容(テキスト) の連想配列を構築し、返します。
function Build-PageBodyTable()
{
    param($RootElement)
    $pageBodyTable = New-Object System.Collections.Hashtable
    $pageBodies = Select-Xml -XPath "//object[@class='BodyContent']" -Xml $RootElement
    foreach ($pageBody in $pageBodies)
    {
        $node = $pageBody.Node
        $pageBodyId = Select-Xml -XPath "./id" -Xml $node | ForEach-Object { $_.Node.InnerText }
        $pageBodyContent = Select-Xml -XPath "./property[@name='body']" -Xml $node | ForEach-Object { $_.Node.InnerText }
        $pageBodyTable[$pageBodyId] = $pageBodyContent
    }
    return $pageBodyTable
}

# Confluence エクスポート XML から
# * ページ ID => ページ情報
# * ページ名 => ページ情報
# の 2 つの連想配列を構築し、返します。
function Build-PageTable()
{
    param($RootElement)
    $pageNameTable = New-Object System.Collections.Hashtable
    $pageIdTable = New-Object System.Collections.Hashtable

    $pages = Select-Xml -XPath "//object[@class='Page']" -Xml $RootElement
    foreach ($page in $pages)
    {
        $pageNode = $page.Node
        $pageName = Select-Xml -XPath "./property[@name='title']" -Xml $pageNode | ForEach-Object { $_.Node.InnerText }
        $pageId = Select-Xml -XPath "./id" -Xml $pageNode | ForEach-Object { $_.Node.InnerText }
        $pageVersion = [int]$(Select-Xml -XPath "./property[@name='version']" -Xml $pageNode | ForEach-Object { $_.Node.InnerText })
        $pageBodyId = Select-Xml -XPath "./collection[@name='bodyContents']/element/id" -Xml $PageNode | ForEach-Object { $_.Node.InnerText }
        $pageChildrenIds = Select-Xml -XPath "./collection[@name='children']/element[@class='Page']/id" -Xml $PageNode | ForEach-Object { $_.Node.InnerText }
        
        $lastPageVersion = 0
        if ($pageNameTable.ContainsKey($pageName))
        {
            $lastPageVersion = $pageNameTable[$pageName].Version
        }

        if ($pageVersion -gt $lastPageVersion)
        {
            $pageInfo = @{
                Id = $pageId
                Name = $pageName
                Version = $pageVersion
                BodyId = $pageBodyId
                ChildrenIds = $pageChildrenIds
            }
            $pageNameTable[$pageName] = $pageInfo
            $pageIdTable[$pageId] = $pageInfo
            Out-LogInfo "[Add] $pageName (ID=$pageId, Version=$pageVersion)"
        }
    }

    return @($pageNameTable, $pageIdTable)
}

# 指定したラベル名を持つラベルの ID を返します。
function Get-LabelId()
{
    param($RootElement, [string]$LabelName)
    return $RootElement | `
        Select-Xml -XPath "//object[@class='Label']/property[@name='name' and text()='$LabelName']/../id" | `
        ForEach-Object { $_.Node.InnerText }
}

# 指定したラベル ID を含むページ ID の一覧を返します。
function Get-LabelContainsPageIds()
{
    param($RootElement, [string]$LabelId)
    return $RootElement | `
        Select-Xml -XPath "//object[@class='Labelling']/property[@name='label']/id[text()='$LabelId']/../../property[@name='content']/id" | `
        ForEach-Object { $_.Node.InnerText }
}

## main ####

function Out-LogInfo()
{
    param([string]$Message)
    [IO.File]::AppendAllText($LogFile, "$Message`r`n", [Text.Encoding]::UTF8)
}

# ログ書き出し先の作成
if ([string]::IsNullOrEmpty($LogDir))
{
    $LogDir = "$TargetRoot\Logs"
}
if (-not(Test-Path $LogDir))
{
    mkdir $LogDir | Out-Null
}
$LogFile = "$LogDir\Remove-FilteredPageFiles.log"

# Confluence XML をインポート
Out-LogInfo "[START] Loading confluence exported XML"
$rootElement = New-Object System.Xml.XmlDocument
$rootElement.Load((Convert-Path $XmlPath))
Out-LogInfo "[FINISH] Loading confluence exported XML"

# 指定したラベルの ID を取得
$labelId = Get-LabelId -RootElement $rootElement -LabelName $FilterLabelName
Out-LogInfo "Label ID: $labelId"

# 指定したラベルを含むページ ID のリストを取得
$labelContainsPageIds = Get-LabelContainsPageIds -RootElement $rootElement -LabelId $labelId
Out-LogInfo ("Label contains page: {0}" -f $($labelContainsPageIds -join ", "))

# XML からページ一覧の情報を構築
Out-LogInfo "[START] Build page information"
$pageBodyTable = Build-PageBodyTable -RootElement $rootElement    
$pageNameTable, $pageIdTable = Build-PageTable -RootElement $rootElement
Out-LogInfo "[FINISH] Build page information"

# 指定したラベルを含むページの子孫ページをすべて取得 
Out-LogInfo "[START] Build descendants pages"
$pageIdSet = New-Object System.Collections.Hashtable
foreach ($pageId in $labelContainsPageIds)
{
    Build-DescendantsPageIds -PageId $pageId -PageIdSet $pageIdSet -PageIdTable $pageIdTable
}
Out-LogInfo "[FINISH] Build descendants pages"

# ラベルが付いた子孫ページ(保存対象となったページ)が参照しているすべての添付ファイルについて、その添付先のフォルダを保護対象にする
# 保護対象となったフォルダは削除されず残るので、ドキュメントパッケージに含まれる
$preserveAttachmentSet = New-Object System.Collections.Hashtable
foreach ($presrevePageId in $pageIdSet.Keys)
{
    # ページ本体を取得
    $pageBodyId = $pageIdTable[$presrevePageId].BodyId
    $pageBodyContent = $pageBodyTable[$pageBodyId]
    # ページ本体から添付ファイルを含む部分を正規表現によって抽出
    $pageBodyContent | Select-String "<([^:]+):attachment[^/>]+>.*?</\1:attachment>" -AllMatches | `
        ForEach-Object {
            # ページ本体に含まれる添付ファイルXMLから、別ページの添付ファイルを参照として持つものをリストアップ
            foreach ($attachmentInBody in $_.Matches) {
                $attachment = ([xml]($attachmentInBody -replace "(?:ri|ac):", "")).FirstChild
                $fileName = $attachment.filename
                $pageElem = $attachment.FirstChild
                if ($pageElem -eq $null) {
                    continue
                }
                # 添付ファイル参照先のページ名からページ情報を取得
                $pageName = $pageElem.GetAttribute("content-title")
                $pageInfo = $pageNameTable[$pageName]
                if ($pageInfo -eq $null) {
                    # 現在のページに添付された画像かどうかを確認
                    $preservePageIdRoot = Select-Xml -XPath "//object[@class='Attachment']/property[@name='content']/id[text()='$presrevePageId']" -Xml $rootElement
                    if ($preservePageIdRoot -ne $null) {
                        # 現在のページに添付された画像なので既に削除対象外となっている
                        $containsPreservePage = Select-Xml -XPath "/../../property[@name='fileName' and text()='$fileName']" -Xml $preservePageIdRoot
                        if ($containsPreservePage -eq $null) {
                            Out-LogInfo "Attachment not found: Page=$pageName, FileName=$fileName"
                        }
                        continue
                    }
                }
                $pageId = $pageInfo.Id
                # 添付ファイル参照先のページ(および添付ファイルフォルダ)を削除対象外としてマーク
                if (-not($preserveAttachmentSet.ContainsKey($pageId))) {
                    Out-LogInfo "[Preserve] PageName: $pageName (ID=$pageId)"
                    $preserveAttachmentSet.Add($pageId, $true)
                }
            }
        }
}

# 指定したラベルを含むページと、その子孫ページを "除く "ページと添付ファイルをすべて削除
@(
    "$TargetRoot\contents\Pages",
    "$TargetRoot\contents\Attachments"
) | 
Get-ChildItem | Where-Object {
    $isRemove = $true
    if ($_.BaseName -match "(Page|Attach)_(\d+)")
    {
        $pageType = $Matches[1]
        $pageId   = $Matches[2]
        if (($pageType -eq "Attach") -and ($preserveAttachmentSet.ContainsKey($pageId))) {
            # 添付ファイルで、かつ削除対象外としてマークされている添付ファイル
            $isRemove = $false
        } else {
            # ページもしくは添付ファイルが削除対象外（ラベル付きページの配下にある）
            $isRemove = -not($pageIdSet.ContainsKey($pageId))
        }

        if ($isRemove)
        {
            $pageInfo = $pageIdTable[$pageId]
            $pageName = $pageInfo.Name
            Out-LogInfo "Remove: $_ ($pageName)"    
        }
    }
    if ($DryRun)
    {
        return $false
    }
    return $isRemove
} | Remove-Item -Recurse

Write-Host "Remove-FileredPageFiles finished, Log file location: $LogFile"
