﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
    プロジェクトファイル(.vcxproj,.sln)内のCafeプラットフォームに関する要素を削除します。

    .DESCRIPTION
    プロジェクトファイル(.vcxproj,.sln)内のCafeプラットフォームに関する要素を削除します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>

[CmdletBinding()]
Param(
    # 対象のルートディレクトリを指定します。
    [parameter(
        Mandatory = $true,
        Position = 0,
        ValueFromPipeline = $true)]
    [string[]] $Path,

    # 対象ファイルのパターン(「*.vcxproj」など)を指定します。
    [parameter()]
    [string[]] $Include = ("*.vcxproj", "*.vcxproj.filters", "*.sln")
)

begin {
    # このスクリプトが置かれているディレクトリ
    $scriptDir = Split-Path $MyInvocation.MyCommand.Path -Parent

    # 除外対象の構成の接頭語
    $ExcludeConfiguration = 'Cafe_'

    filter Remove-Element
    {
        [void]$_.Node.ParentNode.RemoveChild($_.Node)
    }

    function Test-ExcludePlatform
    {
        [CmdletBinding()]
        Param(
            # プラットフォームを指定します。
            [parameter(Mandatory = $true)]
            [string[]] $Platforms,

            # プラットフォームを指定します。
            [parameter(Mandatory = $true)]
            [string] $MatchConfiguration,

            # プラットフォームを指定します。
            [parameter(Mandatory = $true)]
            [string] $MatchPlatform
        )

        $MatchConfiguration.StartsWith($ExcludeConfiguration, [StringComparison]::OrdinalIgnoreCase) -or
        $Platforms -contains $MatchPlatform
    }

    <#
        .SYNOPSIS
        プロジェクトファイル(.vcxproj,sln)内の特定のプラットフォームに関する要素を削除します。

        .DESCRIPTION
        プロジェクトファイル(.vcxproj,sln)内の特定のプラットフォームに関する要素を削除します。

        .INPUTS
        なし。

        .OUTPUTS
        なし。
    #>
    function Remove-Platform
    {
        [CmdletBinding()]
        Param(
            # プロジェクトファイル(.vcxproj,.sln)を指定します。
            [parameter(
                Mandatory = $true,
                Position = 0)]
            [string] $ProjFile,

            # 置換先のプロジェクトファイル(.vcxproj)を指定します。
            # 置換元のプロジェクトファイルと同じパスを指定できます。
            [parameter(
                Mandatory = $true,
                Position = 1)]
            [string] $newProjFile,

            # プラットフォームを指定します。
            [parameter(
                Mandatory = $true,
                Position = 2)]
            [string[]] $ExcludePlatforms
        )

        $ext = [IO.Path]::GetExtension($newProjFile)
        $namespace = @{ m = "http://schemas.microsoft.com/developer/msbuild/2003" }

        switch -Regex ($ext.Substring(1)) {
            "^sln$" {
                $sln = get-content $ProjFile -Encoding UTF8

                $buildRegex = '^\s+{[0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12}}\.(\w+)\|(\w+)\.'

                # ハードリンクで作成された場合に、削除しておかないと元も変更してしまうため、一旦削除する。
                if (Test-Path $newProjFile) { Remove-Item $newProjFile }

                $sln |
                Where-Object {
                    ! ( ($_ -match '^\s+(\w+)\|(\w+) = (\w+)\|(\w+)$' -or $_ -match $buildRegex) -and
                        (Test-ExcludePlatform $ExcludePlatforms $Matches[1] $Matches[2]) )
                } |
                Set-Content $newProjFile -Encoding UTF8
            }
            "^(vcxproj|props|filters)$" {
                [xml]$proj = get-content $ProjFile -Encoding UTF8

                $conditionRegex = '''\$\(Configuration\)\|\$\(Platform\)''\s*==\s*''(\w+)\|(\w+)'''

                # 対象外のプラットフォームが使われている箇所、および
                # 対象外のパッケージプラットフォームが使われている箇所を削除
                #   /Project/ItemGroup/ProjectConfiguration 要素(Include属性で判定)
                $rmEles =
                    @(
                        $proj |
                        Select-Xml "/m:Project/m:ItemGroup/m:ProjectConfiguration" -Namespace $namespace |
                        Where-Object {
                            $_.Node.Include -match '(\w+)\|(\w+)' -and (Test-ExcludePlatform $ExcludePlatforms $Matches[1] $Matches[2])
                        }
                    )
                $rmEles | Remove-Element

                #   全ての Condition 属性を持つ要素(Condition属性で判定)
                $rmEles =
                    @(
                        $proj |
                        Select-Xml "//m:*[@Condition]" -Namespace $namespace |
                        Where-Object {
                            $_.Node.Condition -match $conditionRegex -and (Test-ExcludePlatform $ExcludePlatforms $Matches[1] $Matches[2])
                        }
                    )
                $rmEles | Remove-Element

                # ハードリンクで作成された場合に、削除しておかないと元も変更してしまうため、一旦削除する。
                if (Test-Path $newProjFile) { Remove-Item $newProjFile }
                $proj.Save([IO.Path]::Combine($Pwd, $newProjFile))
            }
        }
    }
}
process {
    Get-ChildItem $Path -Recurse -Include $Include |
    ForEach-Object {
        Remove-Platform $_.FullName $_.FullName ("Cafe", "OasisCafe")
    }
}
