<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        UPP イメージを作成する

    .DESCRIPTION
        FirmwareArchiveNx の nfa ファイルから UPP イメージを作成します。
        このときに各プロセスのメタバージョンを以下の方法で設定できます。
         ・SDKのファームウェアバージョンを一括設定 (デフォルト動作)
         ・オプションから一括指定
         ・設定用のCSVファイルでプロセス毎指定

    .EXAMPLE
        Make-UppImage.ps1 -InputNfa (nfa path) -OutputUppPath (upp path)

    .EXAMPLE
        Make-UppImage.ps1 -InputNfa (nfa path) -OutputUppPath (upp path) -ForceVersion (meta version)

    .EXAMPLE
        Make-UppImage.ps1 -InputNfa (nfa path) -OutputUppPath (upp path) -ConfigPath (version config file path)
#>

# [CmdletBinding()]
Param
(
    # The host IP address pattern as RegularExpression
    [parameter(mandatory=$true)][string]
    $InputNfa,

    [parameter(mandatory=$true)][string]
    $OutputUppPath,

    [string]
    $ConfigPath = "",

    [string]
    $ForceVersion = "",

    [string]
    $BuildType = 'Release',

    [string]
    $Platform = 'NX-NXFP2-a64',

    [switch]
    $ForceDefaultVersionToAll,

    [switch]
    $LeaveIntermediateFiles
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\..\..\Modules\Error"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Path"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Utility"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Target"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Yaml"

function GetVersionsFromConfig
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $filepath,

        [Parameter(Mandatory=$true)][ref]
        $hash
    )

    # ファイルの存在を確認する
    if((Test-Path $filepath) -eq $false){
        write-host("ERROR: Not found CSV-File: " + $filepath)
        return ""
    }

    # CSVファイルを読み込んで、ハッシュに変換する
    $content = Import-Csv $filepath -Header 'name', 'version'
    $hash.Value = @{}
    foreach ($cnt in $content.GetEnumerator())
    {
         $hash.Value.Add($cnt.name, $cnt.version)
    }
}

function GetSdkFirmwareVersion
{
    if(Test-Path $script:FimwareVersionXmlPath)
    {
        $xml = [xml](Get-Content $script:FimwareVersionXmlPath)
        [uint32]$majorVersion = $xml.Product.Version.Major
        [uint32]$minorVersion = $xml.Product.Version.Minor
        [uint32]$microVersion = $xml.Product.Version.Micro
        [uint32]$majorRelstep = $xml.Product.Version.MajorRelstep
        [uint32]$minorRelstep = $xml.Product.Version.MinorRelstep

        return GetVersionValueString $majorVersion $minorVersion $microVersion $majorRelstep $minorRelstep
    }
    else
    {
        throw "Cannot parse firmware version."
    }
}

function GetVersionValueString
{
    Param
    (
        [Parameter(Mandatory=$true)][uint32]$majorVersion,
        [Parameter(Mandatory=$true)][uint32]$minorVersion,
        [Parameter(Mandatory=$true)][uint32]$microVersion,
        [Parameter(Mandatory=$true)][uint32]$majorRelstep,
        [Parameter(Mandatory=$true)][uint32]$minorRelstep
    )

    # return ($majorVersion -shl 26) -bor ($minorVersion -shl 20) -bor ($microVersion -shl 16) -bor ($majorRelstep -shl 8) -bor $minorRelstep
    return ($majorVersion * 0x04000000) -bor ($minorVersion * 0x00100000) -bor ($microVersion * 0x00010000) -bor ($majorRelstep * 0x00000100) -bor $minorRelstep
}

function GetTemporaryFolder
{
    $tempPath = [IO.Path]::GetTempPath()
    $wkFolderPath = $null

    Do{
        $wkFolderPath = Join-Path $tempPath ([IO.Path]::GetRandomFileName())
    }while(Test-Path $wkFolderPath)
    $wkFolder = mkdir $wkFolderPath
    return $wkFolderPath
}

function ForceDeleteFolder
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $directorypath
    )

    Remove-Item -path $directorypath -recurse -force
}

function CheckPath
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $filepath
    )

    if(-not(Test-Path $filepath))
    {
        Write-Host "File is not found. ($filepath)"
        exit
    }
}

function CheckSystemProcessName
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $name
    )

    if(CheckSystemUpdateMetaName $name)
    {
        return $FALSE
    }
    elseif(CheckNameWithDevType $name)
    {
        return $FALSE;
    }

    return $TRUE
}

function CheckSystemUpdateMetaName
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $name
    )

    if($name.StartsWith("SystemUpdate"))
    {
        return $TRUE
    }

    return $FALSE
}

function CheckNameWithDevType
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $name
    )

    if($name.StartsWith("BootImage") -or $name.StartsWith("DevSafeModeUpdater"))
    {
        return $TRUE
    }

    return $FALSE
}

function FindNspNameWithDevKitType
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $findDirectory,

        [Parameter(Mandatory=$true)][string]
        $baseNspName
    )

    if($baseNspName.StartsWith("DevSafeModeUpdater"))
    {
        $selectFiles = Get-ChildItem -LiteralPath $findDirectory | Where-Object BaseName -match "^$baseNspName(K.+)?$"
        return $selectFiles[0].BaseName
    }
    else
    {
        $selectFiles = Get-ChildItem -LiteralPath $findDirectory | Where-Object BaseName -match "^$baseNspName([\-_][^\-_].*)?$"
        return $selectFiles[0].BaseName
    }
}

function EditVersionField
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $path,

        [Parameter(Mandatory=$true)][string]
        $versionString
    )

    if($versionString.ToLower().Contains("0x"))
    {
        $version = [System.Convert]::ToUint32($versionString, 16)
    }
    else
    {
        $version = [System.Convert]::ToUint32($versionString, 10)
    }

    $src = [System.IO.File]::ReadAllBytes($path)

    # $src[0x08] = $version -band 0xFF
    # $src[0x09] = ($version -shr 8) -band 0xFF
    # $src[0x0a] = ($version -shr 16) -band 0xFF
    # $src[0x0b] = ($version -shr 24) -band 0xFF
    $src[0x08] = $version -band 0xFF
    $src[0x09] = ($version / 0x00000100) -band 0xFF
    $src[0x0a] = ($version / 0x00010000) -band 0xFF
    $src[0x0b] = ($version / 0x01000000) -band 0xFF

    [System.IO.File]::WriteAllBytes($path,$src)
}

function GetVersionValue
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $versionString
    )

    if($versionString.ToLower().Contains("0x"))
    {
        return [System.Convert]::ToUint32($versionString, 16)
    }
    else
    {
        return [System.Convert]::ToUint32($versionString, 10)
    }
}

function GetCnmtEntryString
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $nspExtractPath,

        [Parameter(Mandatory=$true)][string]
        $cmntFilePath
    )

    if(-not($cmntFilePath.Contains($nspExtractPath)))
    {
        Write-Host "Cmnt file is not nsp entry. (nsp:{nspExtractPath}, cmnt{cmntFilePath})"
        return ""
    }

    $entryString = $cmntFilePath.SubString($nspExtractPath.Length + 1, $cmntFilePath.Length - $nspExtractPath.Length - 1)
    $entryString = $entryString.Replace("\", "/")

    return $entryString
}

function RenameReplacedNsp
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $nspPath
    )

    $nspDirecotry = Split-Path $nspPath -Parent
    $nspFileBaseName = $(Get-ChildItem $nspPath).BaseName
    $nspFileExtension = $(Get-ChildItem $nspPath).Extension
    $replacedNspFileBaseName = $nspFileBaseName + "_replaced"

    $replacedNspPath = Join-Path $nspDirecotry -ChildPath ($replacedNspFileBaseName + $nspFileExtension)
    MoveNspToOriginalDirectory $nspPath
    Move-Item $replacedNspPath $nspPath
}

function MoveNspToOriginalDirectory
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $nspPath
    )

    Move-Item $nspPath $originalNspDirectory
}

# MEMO: 呼び出しプロセスがログを大量に出力するため表示を抑制するようにしている。
#       ログが必要になった場合は引数 $putLog を $TRUE にして呼び出すようにすること

Set-Variable -Name ExecuteProcessDisableResult -Value -1 -Option constant

function ExecuteProcess
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $exePath,

        [Parameter(Mandatory=$true)][AllowEmptyString()][string]
        $exeArgs,

        [int]
        $returnValue = $ExecuteProcessDisableResult,

        [Boolean]
        $putLog = $FALSE
    )

    $proc = New-Object "System.Diagnostics.Process"
    $psi = New-Object "System.Diagnostics.ProcessStartInfo"

    $psi.FileName = $exePath
    $psi.Arguments = $exeArgs
    $psi.UseShellExecute = $FALSE
    $psi.WindowStyle = [System.Diagnostics.ProcessWindowStyle]::Hidden
    if($putLog  -ne $TRUE)
    {
        $psi.RedirectStandardOutput = $TRUE
    }

    $proc.StartInfo = $psi

    $procOutput = $proc.Start()

    if($putLog  -ne $TRUE)
    {
        $proc.BeginOutputReadLine()
    }

    $proc.WaitForExit()

    if($returnValue -ne $ExecuteProcessDisableResult)
    {
        if($proc.ExitCode -ne $returnValue) {
            Write-Host "error: process execution failed. (path: $exePath, return: $proc.ExitCode)"
            exit
        }
    }
}

function MoveSystemUpdateMeta
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $extractNfaSystemDirectory
    )

    $checkPattern = Join-Path $extractNfaSystemDirectory -ChildPath "SystemUpdate*.nsp"
    $metaFiles = @((Get-Item -path $checkPattern).FullName)
    foreach ($metaFile in $metaFiles)
    {
        Write-Host "move $metaFile"
        MoveNspToOriginalDirectory $metaFile
    }
}

$FimwareVersionXmlPath = "$(Get-NintendoSdkRootPath)/Common\Versions\NXFirmwareVersion.xml"
$MakeFirmwareArchive = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/MakeFirmwareArchive/MakeFirmwareArchive.exe"
# $AuthoringTool = "$(Get-NintendoSdkRootPath)/Programs/Chris/Outputs/x86/Tools/AuthoringTools/AuthoringTool/${BuildType}/AuthoringTool.exe"
$AuthoringTool = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/AuthoringTool/AuthoringTool.exe"
$MakeSystemUpdateMeta ="$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/AnyCPU/Tools/MakeSystemUpdateMeta/MakeSystemUpdateMeta/${BuildType}/MakeSystemUpdateMeta.exe"
$MakeCupArchive = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/AnyCPU/Tools/MakeCupArchive/MakeCupArchive/${BuildType}/MakeCupArchive.exe"
$KeyConfigPath = "$(Get-NintendoSdkRootPath)/Programs/Chris/Sources/Tools/AuthoringTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml"

echo "Make-UppImage.ps1 start"

CheckPath $MakeFirmwareArchive
CheckPath $AuthoringTool
CheckPath $MakeSystemUpdateMeta
CheckPath $MakeCupArchive

if(-not(Test-Path $InputNfa))
{
    Write-Host "error: File of -InputNfa is not found. ($InputNfa)"
    exit
}

if(Test-Path $OutputUppPath)
{
    Remove-Item $OutputUppPath
}

# Tools/CommandLineTools/MakeFirmwareArchive/MakeFirmwareArchive extractnfa -o $CUP_TMP_DIR/extract -i Programs/Eris/Outputs/NX-NXFP2-a64/FirmwareArchives/NX-K3-Hb-Unsigned-Nand-64G-Internal/Develop/NX-K3-Hb-Unsigned-Nand-64G-Internal.nfa
# Tools/CommandLineTools/AuthoringTool/AuthoringTool extract -o $CUP_TMP_DIR/extractnsp/xxxx $CUP_TMP_DIR/extract/SYSTEM/xxxx.nsp --keyconfig Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml
# ($CUP_TMP_DIR/extractnsp/xxxx/[hash].cnmt.nca/.../%Type%_%Id%.cnmt の 0x08～0x0b byteをバイナリエディタで編集する)
# Tools/CommandLineTools/AuthoringTool/AuthoringTool replace $CUP_TMP_DIR/extract/SYSTEM/xxxx.nsp [hash].cnmt.nca D:\\home\\siglo\\temp\\CUP\\20161226\\extractnsp\\xxxx\\[hash].cnmt.nca\\...\\%Type%_%Id%.cnmt -o D:\\home\\siglo\\temp\\CUP\\20161226\\extract\\SYSTEM --keyconfig Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml
# rm $CUP_TMP_DIR/extract/SYSTEM/SystemUpdate.nsp
# Tools/CommandLineTools/MakeSystemUpdateMeta -o $CUP_TMP_DIR/meta.xml --system-meta-id 0x0100000000000816 --system-meta-version 3 --input-directory $CUP_TMP_DIR/extract/SYSTEM
# Tools/CommandLineTools/AuthoringTool/AuthoringTool creatensp -o $CUP_TMP_DIR/extract/SYSTEM/SystemUpdate.nsp --meta $CUP_TMP_DIR/meta.xml --type SystemUpdate --keyconfig Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml
# Tools/CommandLineTools/MakeFirmwareArchive/MakeFirmwareArchive archivenfa -o $CUP_TMP_DIR/modified.nfa -i $CUP_TMP_DIR/extract
# Tools/CommandLineTools/MakeCupArchive -o $CUP_TMP_DIR/cup.nsp -i $CUP_TMP_DIR/modified.nfa --keyconfig Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml

# (process name, version) を入れるハッシュテーブル
$ConfigHash = @{}
[bool] $UseForceVersion = $FALSE
[bool] $UseConfigVersion = $FALSE
[uint32] $ForceVersionValue = 0

if($ForceVersion -ne "")
{
    $UseForceVersion = $TRUE
    $ForceVersionValue = GetVersionValue $ForceVersion
}
elseif($ConfigPath -ne "")
{
    $UseConfigVersion = $TRUE
    GetVersionsFromConfig $ConfigPath ([ref]$ConfigHash)
}
elseif($ForceDefaultVersionToAll)
{
    $UseForceVersion = $TRUE
    $ForceVersionValue = GetSdkFirmwareVersion
}
else
{
    $ForceVersionValue = GetSdkFirmwareVersion
}

$tempDir = GetTemporaryFolder
if($LeaveIntermediateFiles)
{
    echo "temporary directory = $tempDir"
}

$extractNfaDirectory = Join-Path $tempDir "extract"
New-Item $extractNfaDirectory -itemType Directory
$extractNspOfNfaDirectory = Join-Path $tempDir "extractnsp"
New-Item $extractNspOfNfaDirectory -itemType Directory
$originalNspDirectory = Join-Path $tempDir "originalnsp"
New-Item $originalNspDirectory -itemType Directory

$extractNfaSystemDirectory = Join-Path $extractNfaDirectory "SYSTEM"

ExecuteProcess $MakeFirmwareArchive "extractnfa -o $extractNfaDirectory -i $InputNfa"

$systemUpdateMetaVersion = 0

if($UseForceVersion)
{
    # $nspFiles = Get-ChildItem -LiteralPath $extractNfaSystemDirectory | Where-Object Extension -like "nsp"
    $nspFiles = Get-ChildItem -LiteralPath $extractNfaSystemDirectory
    foreach($nspFile in $nspFiles)
    {
        $systemName = $nspFile.BaseName
        $versionString = [string]$ForceVersionValue

        if(CheckSystemProcessName $systemName)
        {
            $extarctNspDirectory = Join-Path $extractNspOfNfaDirectory $systemName
            $systemProcessNspPath = Join-Path $extractNfaSystemDirectory "$systemName.nsp"  # TODO: 付加文字列対応
            CheckPath $systemProcessNspPath
            ExecuteProcess $AuthoringTool "extract -o $extarctNspDirectory $systemProcessNspPath --keyconfig $KeyConfigPath"

            $checkPattern = Join-Path $extarctNspDirectory -ChildPath "*.cnmt.nca" | Join-Path -ChildPath "*" | Join-Path -ChildPath "*.cnmt"
            $cmntFiles = @((Get-Item -path $checkPattern).FullName)
            if($cmntFiles.Count -gt 1)
            {
                Write-Host "warn: cmnt is more than 1 (path: $systemProcessNspPath count : $cmntFiles.Count)"
            }
            $cmntFile = $cmntFiles[0]

            EditVersionField $cmntFile $versionString
            $cmntEntryString = GetCnmtEntryString $extarctNspDirectory $cmntFile

            ExecuteProcess $AuthoringTool "replace $systemProcessNspPath $cmntEntryString $cmntFile -o $extractNfaSystemDirectory --keyconfig $KeyConfigPath"
            RenameReplacedNsp $systemProcessNspPath
        }
        elseif(CheckSystemUpdateMetaName $systemName)
        {
            $systemUpdateMetaVersion = GetVersionValue $versionString
        }
        elseif(CheckNameWithDevType $systemName)
        {
            # BootImagePackage or SaveModeUpdater
            # $withDevKitTypeBaseName = FindNspNameWithDevKitType $extractNfaSystemDirectory $systemBaseNspName

            $extarctNspDirectory = Join-Path $extractNspOfNfaDirectory $systemName
            # $systemProcessNspPath = Join-Path $extractNfaSystemDirectory "$withDevKitTypeBaseName.nsp"
            $systemProcessNspPath = $nspFile.FullName
            CheckPath $systemProcessNspPath
            ExecuteProcess $AuthoringTool "extract -o $extarctNspDirectory $systemProcessNspPath --keyconfig $KeyConfigPath"

            $checkPattern = Join-Path $extarctNspDirectory -ChildPath "*.cnmt.nca" | Join-Path -ChildPath "*" | Join-Path -ChildPath "*.cnmt"
            $cmntFiles = @((Get-Item -path $checkPattern).FullName)
            if($cmntFiles.Count -gt 1)
            {
                Write-Host "warn: cmnt is more than 1 (path: $systemProcessNspPath count : $cmntFiles.Count)"
            }
            $cmntFile = $cmntFiles[0]

            EditVersionField $cmntFile $versionString
            $cmntEntryString = GetCnmtEntryString $extarctNspDirectory $cmntFile

            ExecuteProcess $AuthoringTool "replace $systemProcessNspPath $cmntEntryString $cmntFile -o $extractNfaSystemDirectory --keyconfig $KeyConfigPath"
            RenameReplacedNsp $systemProcessNspPath
        }
    }
}
elseif($UseConfigVersion)
{
    foreach ($key in $ConfigHash.Keys)
    {
        $systemName = $key
        $versionString = $ConfigHash[$key]

        if(CheckSystemProcessName $systemName)
        {
            $extarctNspDirectory = Join-Path $extractNspOfNfaDirectory $systemName
            $systemProcessNspPath = Join-Path $extractNfaSystemDirectory "$systemName.nsp"  # TODO: 付加文字列対応
            CheckPath $systemProcessNspPath
            ExecuteProcess $AuthoringTool "extract -o $extarctNspDirectory $systemProcessNspPath --keyconfig $KeyConfigPath"

            $checkPattern = Join-Path $extarctNspDirectory -ChildPath "*.cnmt.nca" | Join-Path -ChildPath "*" | Join-Path -ChildPath "*.cnmt"
            $cmntFiles = @((Get-Item -path $checkPattern).FullName)
            if($cmntFiles.Count -gt 1)
            {
                Write-Host "warn: cmnt is more than 1 (path: $systemProcessNspPath count : $cmntFiles.Count)"
            }
            $cmntFile = $cmntFiles[0]

            EditVersionField $cmntFile $versionString
            $cmntEntryString = GetCnmtEntryString $extarctNspDirectory $cmntFile

            ExecuteProcess $AuthoringTool "replace $systemProcessNspPath $cmntEntryString $cmntFile -o $extractNfaSystemDirectory --keyconfig $KeyConfigPath"
            RenameReplacedNsp $systemProcessNspPath
        }
        elseif(CheckSystemUpdateMetaName $systemName)
        {
            $systemUpdateMetaVersion = GetVersionValue $versionString
        }
        elseif(CheckNameWithDevType $systemName)
        {
            # BootImagePackage or SaveModeUpdater
            $withDevKitTypeBaseName = FindNspNameWithDevKitType $extractNfaSystemDirectory $systemName

            $extarctNspDirectory = Join-Path $extractNspOfNfaDirectory $systemName
            $systemProcessNspPath = Join-Path $extractNfaSystemDirectory "$withDevKitTypeBaseName.nsp"
            CheckPath $systemProcessNspPath
            ExecuteProcess $AuthoringTool "extract -o $extarctNspDirectory $systemProcessNspPath --keyconfig $KeyConfigPath"

            $checkPattern = Join-Path $extarctNspDirectory -ChildPath "*.cnmt.nca" | Join-Path -ChildPath "*" | Join-Path -ChildPath "*.cnmt"
            $cmntFiles = @((Get-Item -path $checkPattern).FullName)
            if($cmntFiles.Count -gt 1)
            {
                Write-Host "warn: cmnt is more than 1 (path: $systemProcessNspPath count : $cmntFiles.Count)"
            }
            $cmntFile = $cmntFiles[0]

            EditVersionField $cmntFile $versionString
            $cmntEntryString = GetCnmtEntryString $extarctNspDirectory $cmntFile

            ExecuteProcess $AuthoringTool "replace $systemProcessNspPath $cmntEntryString $cmntFile -o $extractNfaSystemDirectory --keyconfig $KeyConfigPath"
            RenameReplacedNsp $systemProcessNspPath
        }
    }
}
else
{
    $systemUpdateMetaVersion = $ForceVersionValue
}

Write-Verbose "SystemUpdater meta version : $systemUpdateMetaVersion"

$systemUpdateMetaPath = Join-Path $extractNfaSystemDirectory "SystemUpdate.nsp"
MoveSystemUpdateMeta $extractNfaSystemDirectory
$systemUpdateMetaXmlPath = Join-Path $tempDir "SystemUpdateMeta.xml"
ExecuteProcess $MakeSystemUpdateMeta "-o $systemUpdateMetaXmlPath  --system-meta-id 0x0100000000000816 --system-meta-version $systemUpdateMetaVersion --input-directory $extractNfaSystemDirectory"

ExecuteProcess $AuthoringTool "creatensp -o $systemUpdateMetaPath --meta $systemUpdateMetaXmlPath --type SystemUpdate --keyconfig $KeyConfigPath"
$modifiedNfaPath = Join-Path $tempDir "modifiedFirmware.nfa"
ExecuteProcess $MakeFirmwareArchive "archivenfa -o $modifiedNfaPath -i $extractNfaDirectory"

ExecuteProcess $MakeCupArchive "-o $OutputUppPath -i $modifiedNfaPath --keyconfig $KeyConfigPath"


if(-not $LeaveIntermediateFiles)
{
    ForceDeleteFolder($tempDir)
}
