<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Make DevKitUpdater.nsp file .

    .DESCRIPTION
        This script makes DevKitUpdaterXXX.nsp .
        Object files are used from Externals/SetupToolBinaries/Objects
#>

# [CmdletBinding()]
Param
(
    # The host IP address pattern as RegularExpression
    [parameter(mandatory=$true)][string]
    $DevKitTypeName,

    [parameter(mandatory=$true)][string]
    $OutputNspPath,

    [parameter(mandatory=$true)][string]
    $UppDirectryPath,

    [parameter(mandatory=$true)][string]
    $NmetaFilePath,

    [parameter(mandatory=$true)][string]
    $DescFilePath,

    [parameter(mandatory=$true)][string]
    $KeyConfigFilePath,

    [string]
    $Platform = 'NX-NXFP2-a64',

    [string]
    $BuildType = 'Release',

    [string]
    $KeyIndex = "",

    [string]
    $KeyGeneration = ""
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\..\..\Modules\Path"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Utility"

# $AuthoringTool = "$(Get-NintendoSdkRootPath)/Programs/Chris/Outputs/x86/Tools/AuthoringTools/AuthoringTool/Release/AuthoringTool.exe"
$AuthoringTool = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/AuthoringTool/AuthoringTool.exe"

function CheckPath
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $filepath
    )

    if(-not(Test-Path $filepath))
    {
        throw "File is not found. ($filepath)"
    }
}

function GetCodeDirectoryPath
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $DevKitTypeName,

        [Parameter(Mandatory=$true)][string]
        $Platform,

        [Parameter(Mandatory=$true)][string]
        $BuildType
    )

    $direcotryPath = "$(Get-NintendoSdkRootPath)\Externals\SetupToolBinaries\Objects\DevKitUpdater\v4\$Platform\$DevKitTypeName\$BuildType\code"
    if(-not(Test-Path $direcotryPath))
    {
        throw "[Error] Program code directory is not found. ($direcotryPath)"
    }

    return $direcotryPath
}

# MEMO: 呼び出しプロセスがログを大量に出力するため表示を抑制するようにしている。
#       ログが必要になった場合は引数 $putLog を $TRUE にして呼び出すようにすること

Set-Variable -Name ExecuteProcessDisableResult -Value -1 -Option constant

function ExecuteProcess
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $exePath,

        [Parameter(Mandatory=$true)][AllowEmptyString()][string]
        $exeArgs,

        [int]
        $returnValue = $ExecuteProcessDisableResult,

        [Boolean]
        $putLog = $FALSE
    )

    $proc = New-Object "System.Diagnostics.Process"
    $psi = New-Object "System.Diagnostics.ProcessStartInfo"

    $psi.FileName = $exePath
    $psi.Arguments = $exeArgs
    $psi.UseShellExecute = $FALSE
    $psi.WindowStyle = [System.Diagnostics.ProcessWindowStyle]::Hidden
    if($putLog  -ne $TRUE)
    {
        $psi.RedirectStandardOutput = $TRUE
    }

    $proc.StartInfo = $psi

    $procOutput = $proc.Start()

    if($putLog  -ne $TRUE)
    {
        $proc.BeginOutputReadLine()
    }

    $proc.WaitForExit()

    if($returnValue -ne $ExecuteProcessDisableResult)
    {
        if($proc.ExitCode -ne $returnValue) {
            Write-Host "[Error] process execution failed. (path: $exePath, return: $proc.ExitCode)"
            exit
        }
    }
}


Write-Host "Make-DevKitUpdaterNsp.ps1 start"

CheckPath $AuthoringTool


if(-not(Test-Path $UppDirectryPath))
{
    throw "[Error] Directory of -UppDirectryPath is not found. ($UppDirectryPath)"
}
if(-not(Test-Path $NmetaFilePath))
{
    throw "[Error] File of -NmetaFilePath is not found. ($NmetaFilePath)"
}
if(-not(Test-Path $DescFilePath))
{
    throw "[Error] File of -DescFilePath is not found. ($DescFilePath)"
}
if(-not(Test-Path $KeyConfigFilePath))
{
    throw "[Error] File of -KeyConfigFilePath is not found. ($KeyConfigFilePath)"
}

if(Test-Path $OutputNspPath)
{
    Remove-Item $OutputNspPath
}

$codeDirecotryPath = GetCodeDirectoryPath $DevKitTypeName $Platform $BuildType
ExecuteProcess $AuthoringTool "creatensp -o $OutputNspPath --save-adf --meta $NmetaFilePath --desc $DescFilePath --type Application --program $codeDirecotryPath $UppDirectryPath --keyconfig $KeyConfigFilePath --keyindex $KeyIndex --keygeneration $KeyGeneration"

Write-Host "Make-DevKitUpdaterNsp.ps1 finished."
