﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Write boot config by QSPI boot

    .DESCRIPTION
        Write boot config by QSPI boot.
#>
[CmdletBinding()]
Param
(
    # The host IP address pattern as RegularExpression
    [string]
    $AddressPattern = '.*',

    # BCT file
    [string]
    $Bct = 'Default', # <BctFile>, Default

    # The target key (PKC) type
    [string]
    $KeyType = 'Auto' # K1, K2, K3, K5, Auto
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}/../../Modules/Error"
Import-Module "${scriptDirectoryPath}/../../Modules/Path"
Import-Module "${scriptDirectoryPath}/../../Modules/HostBridge"
Import-Module "${scriptDirectoryPath}/../../Modules/Utility"
Import-Module "${scriptDirectoryPath}/../../Modules/InitializeTarget"

function Make-QspiBootImage
{
    param
    (
        [string]
        $OutputPath,

        [string]
        $EmmcWriterPath,

        [string]
        $ParameterFilePath,

        [string]
        $ImagePath
    )

    Test-EnsureExistPath $EmmcWriterPath
    Test-EnsureExistPath $ParameterFilePath
    Test-EnsureExistPath $ImagePath

    Get-Content -Encoding Byte -ReadCount 0 $EmmcWriterPath, $ParameterFilePath, $ImagePath | Set-Content $OutputPath -Encoding Byte

    return $OutputPath
}

function Write-Bct
{
    param
    (
        [string]
        $Name,

        [string]
        $ParameterFile
    )

    Write-Output "Write $Name"

    $BctWriterParameter = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/QspiBootTools/$ParameterFile"

    # Make the boot config writer image of writing boot config on QSPI boot.
    $TempPath = [System.IO.Path]::GetTempFileName()
    $QspiBootImage = Make-QspiBootImage -OutputPath $TempPath -EmmcWriterPath $EmmcWriter -ParameterFilePath $BctWriterParameter -ImagePath $BctFile
    Register-EnsureDeleteFile $QspiBootImage

    # Boot Target
    Invoke-CriticalCommand "& `"$StartTarget`" -AddressPattern `"$AddressPattern`""

    # Invoke boot config writer.
    Invoke-CriticalCommand "& `"$InvokeQspiBoot`" -AddressPattern `"$AddressPattern`" -QspiBootImage `"$QspiBootImage`""

    # Wait
    Wait-Seconds 5
}

$InvokeQspiBoot = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Invoke-QspiMode.ps1"
$BctWriterParameter = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/QspiBootTools/params-bct.bin"
$RestartTarget  = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Restart-Target.ps1"
$UpdateHostBridge  = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/Update-HostBridge.ps1"
$StartTarget  = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Start-Target.ps1"

trap [Exception]
{
    Invoke-AllExitingActions
    Write-ErrorRecord $_
    exit 1
}

$TargetAddress = Get-SigleTargetAddress -AddressPattern $AddressPattern
# Update HostBridge
Invoke-CriticalCommand "& `"$UpdateHostBridge`" -AddressPattern `"$AddressPattern`""

if($KeyType -eq 'Auto')
{
    $KeyType = Get-TargetFusedKeyType $TargetAddress
}
switch ( $KeyType )
{
    "K1"
    {
        $EmmcWriter = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/QspiBootTools/EmmcWriter.img"
    }

    "K2"
    {
        $EmmcWriter = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/QspiBootTools/EmmcWriter2ndKey.img"
    }

    "K3"
    {
        $EmmcWriter = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/QspiBootTools/EmmcWriter3rdKey.img"
    }

    "K5"
    {
        $EmmcWriter = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/QspiBootTools/EmmcWriter-K5.img"
    }

    "M1"
    {
        $EmmcWriter = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/QspiBootTools/EmmcWriter-M1.img"
    }

    "M2"
    {
        $EmmcWriter = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/QspiBootTools/EmmcWriter-M2.img"
    }
}

$BctDefault         = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/BootLoader/Normal-$KeyType.bct"
if($Bct -ne "Default")
{
    if(Test-Path $Bct)
    {
        $BctFile = $Bct
    }
    else
    {
        throw "Not found the boot config file: $Bct"
    }
}
else
{
    $BctFile = $BctDefault
}

# Check parameter
Test-ParameterIsNotNullOrEmpty -Name "AddressPattern" -Value $AddressPattern

# Ensure exist boot config
Test-EnsureExistPath $BctFile

Write-Bct "Bct" "params-bct.bin"

# Reboot
Invoke-CriticalCommand "& `"$RestartTarget`" -AddressPattern `"$AddressPattern`""

# Finalize
Invoke-AllExitingActions
Write-Output "Succeeded writing boot config."
