﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        USWriter 経由でジョイコンの Bluetooth 周りの制御を行います

    .DESCRIPTION
        ControlExtCon.exe を利用してジョイコンの Bluetooth 周りの制御を行います。
        ControlExtCon.exe 利用例は以下の通りです。

        * ジョイコンの Bluetooth アドレスを吸い出す
            $ ControlExtCon.exe --port com10 get-device-info-bd_addr
            Try to get device info. port=com10
            40D28AB4E277

        * ペアリングを開始する
            $ ControlExtCon.exe --port com10 reset-pair
            Try to reset. port=com10, page=false, pair=true

        * ペアリング済みの本体に対して Page する
            $ ControlExtCon.exe --port com10 reset-page
            Try to reset. port=com10, page=true, pair=false
#>

#
# Functions
#

function Invoke-Command()
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$comPortNumber,
        [string]$command
    )

    $scriptName = "ControlExtCon.exe"

    pushd $scriptDirectoryPath
    "[Execute] .\$scriptName --port $comPortNumber $command"
    Invoke-CriticalCommand ".\$scriptName --port $comPortNumber $command -Wait"
    popd
}

function GetDeviceInfo()
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$comPortNumber
    )

    $command = "get-device-info-bd_addr"

    Invoke-Command -comPortNumber $comPortNumber -command $command
}

function Pairing()
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$comPortNumber
    )

    $command = "reset-pair"

    Invoke-Command -comPortNumber $comPortNumber -command $command
}

function Page()
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]$comPortNumber
    )

    $command = "reset-page"

    Invoke-Command -comPortNumber $comPortNumber -command $command
}

#
# Import submodules
#

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}/../../Modules/Utility"

#
# Execute commands
#

# ペアリングするコントローラの台数
set ControllerCountMax 2 -option constant

# COM ポート検索用のパターン定義
#  * US Writer のチップが変更された場合、パターンを修正する必要がある。
set ComPortPattern      "'%COM%'" -option constant
set FtdiDeviceIdPattern "'%FTDIBUS\\VID_0403+PID_6001%'" -option constant

# コントローラが接続されている COM ポートの一覧
$foundControllerPorts = @()

# US Writer (特定の FTDI チップが載ったデバイス) を列挙します。
$comPorts = Get-WmiObject -Class Win32_PnPEntity -Filter "Caption LIKE $($ComPortPattern) AND DeviceID LIKE $($FtdiDeviceIdPattern)" | Select-Object -Property Caption
foreach ($portName in $comPorts)
{
    if ($portName -notmatch "COM\d+")
    {
        continue
    }

    # Bluetooth Address を取得し、疎通確認がとれたポートを使用します。
    $portNumber = $matches[0].ToLower()
    GetDeviceInfo -comPortNumber $portNumber
    if ($LastExitCode -eq 0)
    {
        $foundControllerPorts += $portNumber
        if ($foundControllerPorts.Length -eq $ControllerCountMax)
        {
            # 指定された台数見つかったら終了
            break
        }
    }
}

if ($foundControllerPorts.Length -lt $ControllerCountMax)
{
    # コントローラが指定台数見つかりませんでした。
    "failed: $($ControllerCountMax) controllers are not found."
}
else
{
    # ペアリングを開始します。
    foreach ($portNumber in $foundControllerPorts)
    {
        Pairing -comPortNumber $portNumber
    }
}
