﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Kernel Update

    .DESCRIPTION
        Sending a kernel image on serial port, boot it.
#>

param
(
    [string]
    # Path to NxAging config
    $ConfigPath = "c:/home/Default.yml"
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\..\..\Modules\Error"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Path"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Utility"
Import-Module "${scriptDirectoryPath}\..\..\Modules\HostBridge"
Import-Module "${scriptDirectoryPath}\..\..\Modules\BootConfig"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Target"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Yaml"
Import-Module "${scriptDirectoryPath}\..\..\Modules\NxAging"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

# Constants
$KeyType = "K5"
$Platform = "NX-NXFP2-a64"
$ConnectionType = "Hb"

$ControlTarget = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTargetPrivate.exe"
$RestartHostBridge  = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Restart-HostBridge.ps1"
$RunOnTarget = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/RunOnTargetPrivate.exe"
$QspiRecoveryBootToolPath = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/RecoveryBoot.exe"

# Read configuration
$configText = (Get-Content $ConfigPath) -join [System.Environment]::NewLine
$config = (ConvertFrom-Yaml $configText).yaml

if ($config.RunUpdateKernel -eq "false")
{
    Write-Host "Skip update kernel."
    exit 0
}

$BuildType = $config.BuildType
$SignedType = $config.SignedType
$UseProdBoot = $config.UseProdBoot -eq "true"
$UseSdLog = $config.UseSdLog -eq "true"
$UseDummyStarter = $config.UseDummyStarter -eq "true"
$MaxJobCount = $config.MaxJobCount

$TargetList = @()
foreach($node in $config.Target.ChildNodes)
{
    $TargetList += $node.InnerXml
}

# Verify configuration
if ($UseSdLog -and (-not $UseProdBoot))
{
    throw "If you want to use SD logging, you must select ProdBoot."
}

if ($UseDummyStarter -and ($SignedType -eq "Signed"))
{
    throw "If you want to use dummy_starter, you must select Unsigned."
}

if ($TargetList.Count -eq 0)
{
    throw "No valid target is specified."
}

# Functions
function Write-DummyBootConfigForAging()
{
    # TORIAEZU: ブートコンフィグの書き込みは直列じゃないとうまくいかないらしいので、直列実行
    foreach ($target in $TargetList)
    {
        # TORIAEZU: CI で失敗する場合があるので 5 回までリトライする
        for ($i=0; $i -lt 5; $i++)
        {
            try
            {
                # TargetManager が接続中だと takeover できないので切断しておく
                Invoke-Expression "& `"$ControlTarget`" disconnect"

                # 署名済みカーネルを使う場合、署名検証を有効にするために Dummy BootConfig を書く
                Write-Host "Write Dummy BootConfig"
                Write-DummyBootConfig $target
                Write-Host "Done."
                break;
            }
            catch [Exception]
            {
                Write-Host $error

                $num = $i + 1
                Write-Host "Failed to write dummy BootConfig. Retry... ($num/5)"

                # TORIAEZU: ワークアラウンドのため、失敗した場合には HostBridge ごと再起動する
                Invoke-Expression "& `"$RestartHostBridge`" -AddressPattern $target -WaitBoot"
                if($LastExitCode -ne 0)
                {
                    # 再起動に失敗した場合はエラー終了
                    Write-ErrorMessage "Command failed.`n    Command=$Expression`n    ExitCode = $LastExitCode"
                    exit 1
                }
            }
        }
    }
}

function Invoke-RecoveryWriterForAging()
{
    $Timeout = 120
    $RecoveryWriterName = "RecoveryWriter-$KeyType-$ConnectionType-$SignedType"
    $QspiBootImage = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/QspiBootImages/$RecoveryWriterName/$BuildType/$RecoveryWriterName.qspi.img"

    Write-Host "Invoke-RecoveryWriterForAging"

    # 並列に Recovery Writer を実行
    $jobs = @()
    foreach ($target in $TargetList)
    {
        $paramList = @($scriptDirectoryPath, $target, $QspiRecoveryBootToolPath, $QspiBootImage, $RunOnTarget)
        $jobs += Start-Job -ArgumentList $paramList -ScriptBlock {
            param($scriptDirectoryPath, $target, $QspiRecoveryBootToolPath, $QspiBootImage, $RunOnTarget)
            Import-Module "${scriptDirectoryPath}\..\..\Modules\Utility"
            Import-Module "${scriptDirectoryPath}\..\..\Modules\HostBridge"

            # Execute RecoveryWriter
            Write-Host "Start-TargetByHostBridge $target"
            Start-TargetByHostBridge $target

            Write-Host "Execute RecoveryWriter $target"
            Invoke-CriticalCommand "& `"$QspiRecoveryBootToolPath`" --target $target --image `"$QspiBootImage`""

            Wait-Seconds 15

            Write-Host "Restart-TargetByHostBridge $target"
            Restart-TargetByHostBridge $target

            # 安定化のためのウェイト
            Wait-Seconds 15
        }
    }

    # スクリプト完了を待つ
    for ( $i = 0; $i -lt $TargetList.Length; ++$i )
    {
        $target = $TargetList[$i]
        $job = $jobs[$i]

        $done = Wait-Job $job.Id -Timeout $Timeout
        if (-not $done)
        {
            Write-InitializeLog -target $target -log "FAILURE (recovery writer timeout)"
            continue
        }

        $result = Receive-Job $job.Id | Write-Host
        if ($job.State -eq "Failed")
        {
            Write-InitializeLog -target $target -log "FAILURE (recovery writer error)"
            continue
        }

        Write-InitializeLog -target $target -log "SUCCESS"
    }
}

function Invoke-SystemUpdaterForAging()
{
    $Timeout = 300

    $InitialImageName = Get-InitialImageName -useProdBoot ${UseProdBoot} -useSdLog ${UseSdLog} -useDummyStarter ${UseDummyStarter} -keyType ${KeyType} -signedTYpe ${SignedType}
    $InitialImagePath =  "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/InitialImages/$InitialImageName/$BuildType/$InitialImageName.initimg"
    $SystemUpdater = "$(Get-NintendoSdkRootPath)/Programs/Chris/Outputs/$Platform/TargetTools/SystemUpdaterHostFs/$BuildType/SystemUpdaterHostFs.nsp"

    # Target Manager 起動
    Invoke-Expression "& `"${scriptDirectoryPath}\..\..\Start-TargetManager.ps1`" -ForInitialize"

    # 安定化のためのウェイト
    Wait-Seconds 30

    $filteredTargetList = @(Filter-TargetByInitializeLog -targets $TargetList -filter "SUCCESS")

    foreach ($target in $filteredTargetList)
    {
        Invoke-CriticalCommand "& `"$ControlTarget`" register --target $target"
    }

    #
    # SDEV を初期化
    #
    $targetIndex = 0;
    while ($targetIndex -lt $filteredTargetList.Length)
    {
        # 最大同時初期化数: $MaxJobCount
        $targets = @()
        while (($targetIndex -lt $filteredTargetList.Length) -and ($targets.Length -lt $MaxJobCount))
        {
            $targets += $filteredTargetList[$targetIndex]
            $targetIndex++
        }

        # 並列で初期化スクリプトを実行
        $jobs = @()
        foreach ($target in $targets)
        {
            $paramList = @($scriptDirectoryPath, $target, $RunOnTarget, $SystemUpdater, $InitialImagePath)
            $jobs += Start-Job -ArgumentList $paramList -ScriptBlock {
                param($scriptDirectoryPath, $target, $RunOnTarget, $SystemUpdater, $InitialImagePath)
                Import-Module "${scriptDirectoryPath}\..\..\Modules\Utility"

                # Execute SystemUpdater
                Write-Host "Execute SystemUpdater $target"
                $options = "--reset --wait-after-reset 10 --verbose --monitor-serial --failure-timeout 240 --pattern-success-exit `"Succeeded initializing the system.`""
                Invoke-CriticalCommand "& `"$RunOnTarget`" run --target $target --hostbridge $target $options `"$SystemUpdater`" '--' --input `"$InitialImagePath`""

                if ($LastExitCode -ne 0)
                {
                     throw "Command failed.`n"
                }
            }
        }

        # スクリプト完了を待つ
        for ( $i = 0; $i -lt $targets.Length; ++$i )
        {
            $target = $targets[$i]
            $job = $jobs[$i]

            $done = Wait-Job $job.Id -Timeout $Timeout
            if (-not $done)
            {
                Write-InitializeLog -target $target -log "FAILURE (system updater timeout)"
                continue
            }

            $result = Receive-Job $job.Id | Write-Host
            if ($job.State -eq "Failed")
            {
                Write-InitializeLog -target $target -log "FAILURE (system updater error)"
                continue
            }

            Write-InitializeLog -target $target -log "SUCCESS"
        }
    }
}

function Invoke-TargetResetForAging()
{
    $Timeout = 60

    $filteredTargetList = @(Filter-TargetByInitializeLog -targets $TargetList -filter "SUCCESS")

    $jobs = @()
    foreach ($target in $filteredTargetList)
    {
        $paramList = @($ControlTarget, $target)
        $jobs += Start-Job -ArgumentList $paramList -ScriptBlock {
            param($ControlTarget, $target)

            # 製品相当初期化だと TM につながらなくなるので、
            # TM を使わずに再起動できる電源ボタン長押しを使う
            Invoke-Expression "& `"$ControlTarget`" press-power-button-for-sc7 --hold-time 13 --ip-addr $target"
            if ($LastExitCode -ne 0)
            {
                throw "Command failed.`n& `"$ControlTarget`" press-power-button-for-sc7 --hold-time 13 --ip-addr $target`nExitCode = $LastExitCode"
            }
            Start-Sleep -s 3
            Invoke-Expression "& `"$ControlTarget`" press-power-button-for-sc7 --hold-time 1 --ip-addr $target"
            if ($LastExitCode -ne 0)
            {
                throw "Command failed.`n& `"$ControlTarget`" press-power-button-for-sc7 --hold-time 1 --ip-addr $target`nExitCode = $LastExitCode"
            }
        }
    }

    # スクリプト完了を待つ
    for ( $i = 0; $i -lt $filteredTargetList.Length; ++$i )
    {
        $target = $filteredTargetList[$i]
        $job = $jobs[$i]

        $done = Wait-Job $job.Id -Timeout $Timeout
        if (-not $done)
        {
            Write-InitializeLog -target $target -log "FAILURE (target reset)"
            continue
        }

        $result = Receive-Job $job.Id | Write-Host
        if ($job.State -eq "Failed")
        {
            Write-InitializeLog -target $target -log "FAILURE (target reset)"
            continue
        }

        Write-InitializeLog -target $target -log "SUCCESS"
    }
}

Write-Host "Clear result directory."
Clear-ResultDirectory

Write-Host "Update kernel image for aging."

# Signed のみ dummy boot config 書き込み
if ($SignedType -eq "Signed")
{
    Write-Host "Write dummy boot config."
    Write-DummyBootConfigForAging
}

# Recovery Writer の実行 (失敗してもリトライはしない、失敗したターゲットを記録する)
Write-Host "Invoke RecoveryWriter."
Invoke-RecoveryWriterForAging

Wait-seconds 30

# System Updater の実行 (失敗してもリトライはしない、失敗したターゲットを記録する)
Write-Host "Invoke SystemUpdater."
Invoke-SystemUpdaterForAging

# 再起動
Write-Host "Reset target."
Invoke-TargetResetForAging

# ProdMode で Dummy Starter を使った場合は、時間をあけてもう一度リセットすることが必要
if ($UseProdBoot -and $UseDummyStarter)
{
    Wait-Seconds 60
    Write-Host "Reset target."
    Invoke-TargetResetForAging
}

# SIGLO-47686 ワークアラウンド: 再起動失敗に備えて再起動回数を増やす
Wait-Seconds 60
Write-Host "Reset target."
Invoke-TargetResetForAging

Write-Host "Done."
