﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Switch Menu

    .DESCRIPTION
        Sending a kernel image on serial port, boot it.
#>

param
(
    [parameter(Mandatory=$true)]
    [string]
    # Path to NX aging configuration file
    $ConfigPath,

    [parameter(Mandatory=$true)]
    [string]
    # Key name of command list in YAML file
    $Key
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\..\..\Modules\Error"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Path"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Utility"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Target"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Yaml"
Import-Module "${scriptDirectoryPath}\..\..\Modules\NxAging"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

# Read configuration
$configText = (Get-Content $ConfigPath) -join [System.Environment]::NewLine
$config = (ConvertFrom-Yaml $configText).yaml

$BuildType = $config.BuildType
$DefaultMaxJobCount = $config.MaxJobCount

$DefaultTargetList = @()
foreach ($node in $config.Target.ChildNodes)
{
    $DefaultTargetList += $node.InnerXml
}

Write-Host "Go"

$InvalidTargetList = @(Filter-TargetByInitializeLog -targets $DefaultTargetList -filter "FAILURE")
foreach ($target in $InvalidTargetList)
{
    Write-Host "Target ${target} is invalid."
    Write-Host "    $(Read-InitializeLog ${target})"
}

$ValidTargetList = @(Filter-TargetByInitializeLog -targets $DefaultTargetList -filter "SUCCESS")

$CommandList = $config.$Key
if (-not $CommandList)
{
    Write-Host "`"$Key`" has no command list."
    exit 0
}

# Key specific configuration
# TODO: Setup / Teardown 用のスクリプトを用意して、リファクタ
$WriteInitializeLog = $false
$ErrorByCommandFailure = $false
if ($Key -eq "Setup")
{
    $WriteInitializeLog = $true
    $ErrorByCommandFailure = $false
}
if ($Key -eq "Teardown")
{
    $WriteInitializeLog = $false
    $ErrorByCommandFailure = $true
}

# Verify configuration
if ($ValidTargetList.Count -eq 0)
{
    throw "No valid target is specified."
}

# Constants
$Platform = "NX-NXFP2-a64"

$RunOnTarget = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/RunOnTarget.exe"
$ControlTarget = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe"
$ControlTargetPrivate = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTargetPrivate.exe"
$DevMenuCommand = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/${Platform}/TargetTools/DevMenuCommand/${BuildType}/DevMenuCommand.nsp"
$DevMenuCommandSystem = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/${Platform}/TargetTools/DevMenuCommandSystem/${BuildType}/DevMenuCommandSystem.nsp"
$SettingsManager = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/${Platform}/TargetTools/SettingsManager/${BuildType}/SettingsManager.nsp"

# Functions
function Invoke-Script()
{
    param
    (
        [parameter(Mandatory=$true)]
        [string]
        $Command,

        [parameter(Mandatory=$true)]
        [string[]]
        $TargetList,

        [int]
        $MaxJobCount,

        [int]
        $Timeout = 120
    )

    # DevMenuCommand を実行
    $targetIndex = 0;
    while ($targetIndex -lt $TargetList.Length)
    {
        # 最大同時初期化数: $MaxJobCount
        $targets = @()
        while (($targetIndex -lt $TargetList.Length) -and ($targets.Length -lt $MaxJobCount))
        {
            $targets += $TargetList[$targetIndex]
            $targetIndex++
        }

        # 並列で RunOnTarget を実行
        $jobs = @()
        foreach ($target in $targets)
        {
            $paramList = @($Command, $target, $RunOnTarget, $ControlTarget, $ControlTargetPrivate, $DevMenuCommand, $DevMenuCommandSystem, $SettingsManager)
            $jobs += Start-Job -ArgumentList $paramList -ScriptBlock {
                param($command, $target, $RunOnTarget, $ControlTarget, $ControlTargetPrivate, $DevMenuCommand, $DevMenuCommandSystem, $SettingsManager)

                # 設定ファイルに記述したコマンドで展開可能な環境変数の設定
                Set-Item env:TARGET $target
                Set-Item env:RUN_ON_TARGET $RunOnTarget
                Set-Item env:CONTROL_TARGET $ControlTarget
                Set-Item env:CONTROL_TARGET_PRIVATE $ControlTargetPrivate
                Set-Item env:DEV_MENU_COMMAND $DevMenuCommand
                Set-Item env:DEV_MENU_COMMAND_SYSTEM $DevMenuCommandSystem
                Set-Item env:SETTINGS_MANAGER $SettingsManager

                $LastExitCode = 0
                $cmd = [System.Environment]::ExpandEnvironmentVariables($command)
                Write-Host $cmd
                Invoke-Expression "& $cmd"

                if ($LastExitCode -ne 0)
                {
                    throw "Command failed (LastExitCode = $LastExitCode).`n& $cmd"
                }
            }
        }

        # スクリプト完了を待つ
        $commandFailure = $false
        for ( $i = 0; $i -lt $targets.Length; ++$i )
        {
            $target = $targets[$i]
            $job = $jobs[$i]

            $done = Wait-Job $job.Id -Timeout $Timeout
            if (-not $done)
            {
                if ($WriteInitializeLog)
                {
                    Write-InitializeLog -target $target -log "FAILURE (`"$cmd`" timeout)"
                }
                $commandFailure = $true
                continue
            }

            $result = Receive-Job $job.Id | Write-Host
            if ($job.State -eq "Failed")
            {
                if ($WriteInitializeLog)
                {
                    Write-InitializeLog -target $target -log "FAILURE (`"$cmd`" updater error)"
                }
                $commandFailure = $true
                continue
            }
        }

        if ($ErrorByCommandFailure -and $commandFailure)
        {
            throw "Invoke-Scripts.ps1 failed by command failure."
        }
    }
}

# コマンドリストの実行
foreach($command in $CommandList.ChildNodes)
{
    $cmd = $command.Command

    # コマンドを実行するターゲットを取得
    $targetList = $ValidTargetList
    if ($command.Target -ne $null)
    {
        $targetList = @()
        foreach($node in $command.Target.ChildNodes)
        {
            $targetList += $node.InnerXml
        }
    }

    # 初期化などに失敗したターゲットは取り除く
    $targetList = @(Filter-TargetByInitializeLog -targets $targetList -filter "SUCCESS")

    # コマンドの最大並列実行数を取得
    $maxJobCount = $DefaultMaxJobCount
    if ($command.MaxJobCount -ne $null)
    {
        $maxJobCount = $command.MaxJobCount
    }

    # 安定化のため、コマンドごとに Target Manager を再起動
    Write-Host "Restart Target Manager."
    Invoke-Expression "& `"${scriptDirectoryPath}\..\..\Stop-TargetManager.ps1`""
    Invoke-Expression "& `"${scriptDirectoryPath}\..\..\Start-TargetManager.ps1`""
    foreach ($target in $targetList)
    {
        Invoke-CriticalCommand "& `"$ControlTarget`" register --target $target"
    }

    # コマンドの実行
    Write-Host "Invoke-Script `"$cmd`" target: $targetList maxJobCount: $maxJobCount"
    Invoke-Script $cmd $targetList $maxJobCount
}

Write-Host "Done."
