﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Utility Module

    .DESCRIPTION
        This file defines functions for utility operations
#>

$moduleFilePath = $MyInvocation.MyCommand.Path
$modulePath     = [System.IO.Path]::GetDirectoryName($moduleFilePath)
$moduleRootPath = [System.IO.Path]::GetDirectoryName($modulePath)

Import-Module "${moduleRootPath}\Path"

function Test-EnsureExistPath
{
    param
    (
        [string]
        #Test path
        $Path
    )
    
    if([String]::IsNullOrEmpty($Path))
    {
        throw "File not found: $Path";
    }
    
    if ( (Test-Path -path $Path) -eq $false  )
    {
        throw "File not found: $Path";
    }
}
Export-ModuleMember -Function Test-EnsureExistPath

function Invoke-CriticalCommand()
{
    param
    (
        [string]
        # Invoked expression
        $Expression
    )

    #Write-Output "$Expression"
    Invoke-Expression $Expression
    if($LastExitCode -ne 0)
    {
        throw "Command failed.`n    Command=$Expression`n    ExitCode = $LastExitCode"
    }
}
Export-ModuleMember -Function Invoke-CriticalCommand

$global:ExitingActions = New-Object 'System.Collections.Generic.List[ScriptBlock]'
Export-ModuleMember -Var $global:ExitingActions

function Invoke-AllExitingActions
{
    foreach($action in $global:ExitingActions)
    {
        $action.Invoke()
    }
}
Export-ModuleMember -Function Invoke-AllExitingActions

function Register-ExitingEvnet
{
    param
    (
        [ScriptBlock]
        $Action
    )
    
    $global:ExitingActions.Add($Action)
}
Export-ModuleMember -Function Register-ExitingEvnet

function Register-EnsureDeleteFile
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $Target
    )
    
    Register-ExitingEvnet {
        if (Test-Path $script:Target) 
        {
            Remove-Item $script:Target
        }        
    }.GetNewClosure()
}
Export-ModuleMember -Function Register-EnsureDeleteFile

function Test-ParameterIsNotNullOrEmpty
{
    param
    (
        [string]
        $Name,

        [string]
        $Value
    )
    
    if([String]::IsNullOrEmpty($Value))
    {
        throw "Parameter is null: name=$Name, value=$Value"
    }
}
Export-ModuleMember -Function Test-ParameterIsNotNullOrEmpty

<#
    .SYNOPSIS
        Wait few seconds with message
#>
function Wait-Seconds
{
    param
    (
        [Int]
        $RetryCount = 10,
        
        [string]
        $Message = "Wait..."
    )
        
    for($i=1; $i -le $RetryCount; $i++)
    {
        Write-Output "$Message ($i/$RetryCount)";
        Start-Sleep -s 1
    }
}
Export-ModuleMember -Function Wait-Seconds

function Stop-ProcessByName
{
    param
    (
        [string]
        $Name = ""
    )
    
    Test-ParameterIsNotNullOrEmpty -Name "Name" -Value $Name 

    Stop-Process -Name $Name -ErrorAction SilentlyContinue
}
Export-ModuleMember -Function Stop-ProcessByName

function Close-ProgramGracefullyByName
{
    param
    (
        [string]
        $Name = ""
    )
    
    Test-ParameterIsNotNullOrEmpty -Name "Name" -Value $Name

    $processes = (Get-Process)
    foreach($process in $processes)
    {
        if($process.Name -eq $Name)
        {
            $ret = $process.CloseMainWindow()
            if( !$ret )
            {
                return $FALSE
            }
        }
    }

    # CloseMainWindow() は実際に終了したかを確認しないので確認する
    $RetryCount = 5
    while($RetryCount -gt 0 )
    {
        $count = 0
        $processes = (Get-Process)
        foreach($process in $processes)
        {
            if($process.Name -eq $Name)
            {
                $count++
            }
        }
        if($count -eq 0)
        {
            break
        }
        $RetryCount--
        Start-Sleep -s 1
    }
    if( $RetryCount -eq 0 )
    {
        return $FALSE
    }

    return $TRUE
}
Export-ModuleMember -Function Close-ProgramGracefullyByName

<#
    .SYNOPSIS
        Kill processes for system initialize
#>
function Stop-ProcessesForSystemInitialize
{
    param
    (
        [switch]$SkipStopTargetManager
    )

    if(!$SkipStopTargetManager)
    {
        # TargetManager がタスクトレイに入っているとうまく終了できないので、一旦ウィンドウを表示させる
        $TargetManagerPath = "$(Get-NintendoSdkRootPath)\Externals\Oasis\bin\NintendoTargetManager.exe"
        $TargetManagerProcess = Get-Process | where {$_.Name.Equals("NintendoTargetManager")}
        if (($TargetManagerProcess -ne $null) -and ((Test-Path -path $TargetManagerPath) -eq $true))
        {
            Start-Process $TargetManagerPath
            Start-Sleep -s 1
        }

        $ret = Close-ProgramGracefullyByName -Name "NintendoTargetManager"
        if( !$ret )
        {
            Write-Host "Failed to terminate NintendoTargetManager. So, kill NintendoTargetManager process."
            Stop-ProcessByName -Name NintendoTargetManager
        }    
    }

    Stop-ProcessByName -Name RunOnTargetPrivate
    Stop-ProcessByName -Name RecoveryBoot
    Stop-ProcessByName -Name UpdateHostBridge
    Stop-ProcessByName -Name RunOnTarget
    Stop-ProcessByName -Name ControlTarget
    Stop-ProcessByName -Name RunOnTargetFromNand
    Stop-ProcessByName -Name RunSystemInitializerManu
    Stop-ProcessByName -Name RequestToWsIssuer
}
Export-ModuleMember -Function Stop-ProcessesForSystemInitialize

function Write-Zero
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $FilePath,

        [Parameter(Mandatory=$true)]
        [Int]
        $Start,
        
        [Parameter(Mandatory=$true)]
        [Int]
        $Size
    )

    $code = @"
using System;
using System.IO;

namespace FileUtil
{
    public class FileUtilWriteZero
    {
        public static void WriteZero(string filePath, int start, int size)
        {
            using (var fileStream = File.OpenWrite(filePath))
            {
                fileStream.Seek(start, SeekOrigin.Begin);
                byte[] bytes = new byte[size];
                fileStream.Write(bytes, 0, size);
            }
        }
    }
}
"@

    Add-Type -Language CSharpVersion3 $code;

    [FileUtil.FileUtilWriteZero]::WriteZero($FilePath, $Start, $Size);
}
Export-ModuleMember -Function Write-Zero

function New-TruncatedFile
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $Source,

        [Parameter(Mandatory=$true)]
        [string]
        $Output,
        
        [Parameter(Mandatory=$true)]
        [Int]
        $Size
    )

    $code = @"
using System;
using System.IO;

namespace FileUtil
{
    public class FileUtilCreateTruncatedFile
    {
        public static void CreateTruncatedFile(string source, string output, int size)
        {
            int newSize = Math.Min(size, (int)(new FileInfo(source).Length));
            byte[] bytes = new byte[newSize];
            using(var sourceStream = File.OpenRead(source))
            {
                sourceStream.Read(bytes, 0, newSize);                
            }
            using (var outputStream = File.OpenWrite(output))
            {
                outputStream.Write(bytes, 0, newSize);
            }
        }
    }
}
"@

    Add-Type -Language CSharpVersion3 $code;

    [FileUtil.FileUtilCreateTruncatedFile]::CreateTruncatedFile($Source, $Output, $Size);
}
Export-ModuleMember -Function New-TruncatedFile

function Get-NameAndHash
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $FileName
    )
    
    Write-Output "$($(Get-Item $FileName).Name) $($(Get-FileHash $FileName).Hash)" 
}
Export-ModuleMember -Function Get-NameAndHash

function Get-NameAndHashTruncated
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $FileName,
        
        [Parameter(Mandatory=$true)]
        [Int]
        $TruncatedSize        
    )
    
    $tempfile = [System.IO.Path]::GetTempFileName()
    New-TruncatedFile -Source $FileName -Output $tempfile -Size $TruncatedSize
    Write-Output "$($(Get-Item $FileName).Name) $($(Get-FileHash $tempfile).Hash)" 
    rm $tempfile 
}
Export-ModuleMember -Function Get-NameAndHashTruncated

function Import-CsScripts
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $ModuleName,

        [Parameter(Mandatory=$false)]
        [string]
        $ReferencedAssemblies = ""
    )

    $csScriptFilePath = "$(Get-NintendoSdkRootPath)\Integrate\Sources\Tools\CsScripts\CsScripts\$ModuleName.cs"
    
    if($ReferencedAssemblies -eq "")
    {
        Add-Type -Path "$csScriptFilePath"
    }
    else
    {
        Add-Type -Path "$csScriptFilePath" -ReferencedAssemblies $ReferencedAssemblies
    }
}
Export-ModuleMember -Function Import-CsScripts