﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        TeamCity Module

    .DESCRIPTION
        This file defines functions for TeamCity operations
#>


function Get-RestApiUrl([string]$url)
{
    return "${url}/httpAuth/app/rest"
}

function Get-FlattenedLocator([hashtable]$locator)
{
    $list = New-Object "System.Collections.Generic.List[string]"

    $locator.Keys | % {
        $key   = $_
        $value = $locator[$key]
        if ($value -is [hashtable])
        {
            $list.Add("${key}:($(Get-FlattenedLocator $value))")
        }
        else
        {
            $list.Add("${key}:${value}")
        }
    }

    return $list -join ","
}

function Get-RequestUrl([string]$url, [hashtable]$locator)
{
    if ($locator)
    {
        return "${url}?locator=$(Get-FlattenedLocator $locator)"
    }

    return $url
}

function Invoke-RestApi
{
    param
    (
        [string]     $url,
        [string]     $userName,
        [string]     $password,
        [scriptblock]$contentWriter
    )

    $request = [System.Net.WebRequest]::Create($url)
    $request.PreAuthenticate = $true
    $request.Credentials = New-Object "System.Net.NetworkCredential"
    $request.Credentials.UserName = $userName
    $request.Credentials.Password = $password
    & $contentWriter $request
    try
    {
        $response = $request.GetResponse()
        $stream   = $response.GetResponseStream()
        $reader   = New-Object "System.IO.StreamReader" $stream
        return $reader.ReadToEnd()
    }
    finally
    {
        if ($reader)   { $reader.Dispose() }
        if ($stream)   { $stream.Dispose() }
        if ($response) { $response.Close() }
    }
}

function Invoke-RestApiUsingGet
{
    param
    (
        [Parameter(Mandatory=$true)][string]$url,
        [Parameter(Mandatory=$true)][string]$userName,
        [Parameter(Mandatory=$true)][string]$password
    )

    $contentWriter = {
        param ([System.Net.WebRequest]$request)

        $request.Method = "GET"
    }

    return (Invoke-RestApi $url $userName $password $contentWriter)
}

function Invoke-RestApiUsingPut
{
    param
    (
        [Parameter(Mandatory=$true)][string]$url,
        [Parameter(Mandatory=$true)][string]$userName,
        [Parameter(Mandatory=$true)][string]$password,
        [Parameter(Mandatory=$true)][string]$contentType,
        [Parameter(Mandatory=$true)][string]$body
    )

    $contentWriter = {
        param ([System.Net.WebRequest]$request)

        $request.Method = "PUT"
        $request.ContentType = $contentType

        $requestStream = $request.GetRequestStream()
        $data = [system.text.encoding]::ascii.GetBytes($body)
        $requestStream.Write($data, 0, $data.Length)
        $requestStream.Close()
    }

    return (Invoke-RestApi $url $userName $password $contentWriter)
}

function New-ServiceMessageArguments
{
    return New-Object "System.Collections.Generic.Dictionary[string,string]"
}

function ConvertTo-ServiceMessageString([string]$str)
{
    $str = $str -creplace "\|", "||"
    $str = $str -creplace "'",  "|'"
    $str = $str -creplace "\n", "|n"
    $str = $str -creplace "\r", "|r"
    $str = $str -creplace "\[", "|["
    $str = $str -creplace "\]", "|]"
    return $str
}

function Get-ServiceMessage([string]$massageName, [object]$arguments)
{
    $builder = New-Object "System.Text.StringBuilder"
    [void]$builder.Append("##teamcity[")
    [void]$builder.Append($massageName)
    if ( $arguments -is [System.Collections.Generic.Dictionary[string,string]] )
    {
        $arguments.Keys | % {
            $key   = $_
            $value = $arguments[$_]
            [void]$builder.Append(" ")
            [void]$builder.Append($key)
            [void]$builder.Append("='")
            [void]$builder.Append((ConvertTo-ServiceMessageString $value))
            [void]$builder.Append("'")
        }
    }
    else
    {
        [void]$builder.Append(" ")
        [void]$builder.Append("'")
        [void]$builder.Append((ConvertTo-ServiceMessageString $arguments))
        [void]$builder.Append("'")
    }
    [void]$builder.Append("]")
    return $builder.ToString()
}

<#
    .SYNOPSIS
        Returns the detail of a change

    .OUTPUTS
        XmlDocument instance
#>
function Get-TeamCityChangeDetail
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # CI server url
        $Url,

        [Parameter(Mandatory=$true)]
        [string]
        # User name
        $UserName,

        [Parameter(Mandatory=$true)]
        [string]
        # Password
        $Password,

        [string]
        # Change Id
        $ChangeId
    )

    $requestUrl = "$(Get-RestApiUrl $Url)/changes/id:${ChangeId}"

    return [xml](Invoke-RestApiUsingGet $requestUrl $UserName $Password)
}
Export-ModuleMember -Function Get-TeamCityChangeDetail

<#
    .SYNOPSIS
        Returns a list of changes

    .OUTPUTS
        XmlDocument instance
#>
function Get-TeamCityChangeList
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # CI server url
        $Url,

        [Parameter(Mandatory=$true)]
        [string]
        # User name
        $UserName,

        [Parameter(Mandatory=$true)]
        [string]
        # Password
        $Password,

        [hashtable]
        # Specify entities to filter
        $Locator
    )

    $requestUrl = Get-RequestUrl "$(Get-RestApiUrl $Url)/changes" $Locator

    return [xml](Invoke-RestApiUsingGet $requestUrl $UserName $Password)
}
Export-ModuleMember -Function Get-TeamCityChangeList

<#
    .SYNOPSIS
        Enter an testing session for TeamCity
#>
function Enter-TestingSessionForTeamCity
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # Test suite name
        $SuiteName,

        [Parameter(Mandatory=$true)]
        [string]
        # Test name
        $TestName
    )

    $arguments = (New-ServiceMessageArguments)
    $arguments["name"] = $SuiteName
    Write-Host (Get-ServiceMessage "testSuiteStarted" $arguments)

    $arguments = (New-ServiceMessageArguments)
    $arguments["name"]                  = $TestName
    $arguments["captureStandardOutput"] = "true"
    Write-Host (Get-ServiceMessage "testStarted" $arguments)
}
Export-ModuleMember -Function Enter-TestingSessionForTeamCity

<#
    .SYNOPSIS
        Exit an testing session for TeamCity
#>
function Exit-TestingSessionForTeamCity
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # Test suite name
        $SuiteName,

        [Parameter(Mandatory=$true)]
        [string]
        # Test name
        $TestName,

        [string]
        # Set the test failed and specify the error message
        $ErrorMessage
    )

    if ( $ErrorMessage )
    {
        $arguments = (New-ServiceMessageArguments)
        $arguments["name"]    = $TestName
        $arguments["message"] = $ErrorMessage
        Write-Host (Get-ServiceMessage "testFailed" $arguments)
    }

    $arguments = (New-ServiceMessageArguments)
    $arguments["name"] = $TestName
    Write-Host (Get-ServiceMessage "testFinished" $arguments)

    $arguments = (New-ServiceMessageArguments)
    $arguments["name"] = $SuiteName
    Write-Host (Get-ServiceMessage "testSuiteFinished" $arguments)
}
Export-ModuleMember -Function Exit-TestingSessionForTeamCity

<#
    .SYNOPSIS
        Set an environment variable for TeamCity
#>
function Set-EnvironmentVariableForTeamCity
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # Variable name
        $Name,

        [Parameter(Mandatory=$true)]
        [string]
        # Variable value
        $Value
    )

    $arguments = (New-ServiceMessageArguments)
    $arguments["name"]  = "env.${Name}"
    $arguments["value"] = $Value

    Write-Host (Get-ServiceMessage "setParameter" $arguments)
}
Export-ModuleMember -Function Set-EnvironmentVariableForTeamCity


<#
    .SYNOPSIS
        Get TeamCity build log
#>
function Get-TeamCityBuildLog
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # CI server url
        $Url,

        [Parameter(Mandatory=$true)]
        [string]
        # User name
        $UserName,

        [Parameter(Mandatory=$true)]
        [string]
        # Password
        $Password,

        [Parameter(Mandatory=$true)]
        [string]
        # BuildId
        $BuildId
    )

    $requestUrl = "${Url}/downloadBuildLog.html?buildId=${BuildId}"

    return (Invoke-RestApiUsingGet $requestUrl $UserName $Password)
}
Export-ModuleMember -Function Get-TeamCityBuildLog


<#
    .SYNOPSIS
        Get TeamCity users
#>
function Get-TeamCityUsers
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # CI server url
        $Url,

        [Parameter(Mandatory=$true)]
        [string]
        # User name
        $UserName,

        [Parameter(Mandatory=$true)]
        [string]
        # Password
        $Password
    )

    $requestUrl = "$(Get-RestApiUrl $Url)/users"

    return [Xml](Invoke-RestApiUsingGet $requestUrl $UserName $Password)
}
Export-ModuleMember -Function Get-TeamCityUsers


<#
    .SYNOPSIS
        Get TeamCity user field
#>
function Get-TeamCityUserField
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # CI server url
        $Url,

        [Parameter(Mandatory=$true)]
        [string]
        # User name
        $UserName,

        [Parameter(Mandatory=$true)]
        [string]
        # Password
        $Password,

        [Parameter(Mandatory=$true)]
        [string]
        # User locator
        $UserLocator,

        [Parameter(Mandatory=$true)]
        [string]
        # Field name
        $FieldName
    )

    $requestUrl = "$(Get-RestApiUrl $Url)/users/${UserLocator}/${FieldName}"

    return (Invoke-RestApiUsingGet $requestUrl $UserName $Password)
}
Export-ModuleMember -Function Get-TeamCityUserField


<#
    .SYNOPSIS
        Get TeamCity user property
#>
function Get-TeamCityUserProperty
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # CI server url
        $Url,

        [Parameter(Mandatory=$true)]
        [string]
        # User name
        $UserName,

        [Parameter(Mandatory=$true)]
        [string]
        # Password
        $Password,

        [Parameter(Mandatory=$true)]
        [string]
        # User locator
        $UserLocator,

        [Parameter(Mandatory=$true)]
        [string]
        # Property name
        $PropertyName
    )

    $requestUrl = "$(Get-RestApiUrl $Url)/users/${UserLocator}/properties/${PropertyName}"

    try
    {
        return (Invoke-RestApiUsingGet $requestUrl $UserName $Password)
    }
    catch [System.Net.WebException]
    {
        if ([System.Net.WebExceptionStatus]::ProtocolError -eq $_.Exception.Status)
        {
            if (404 -eq [int]$_.Exception.Response.StatusCode)
            {
                return $null
            }
        }
        throw
    }
}
Export-ModuleMember -Function Get-TeamCityUserProperty


<#
    .SYNOPSIS
        Set TeamCity user property
#>
function Set-TeamCityUserProperty
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # CI server url
        $Url,

        [Parameter(Mandatory=$true)]
        [string]
        # User name
        $UserName,

        [Parameter(Mandatory=$true)]
        [string]
        # Password
        $Password,

        [Parameter(Mandatory=$true)]
        [string]
        # User locator
        $UserLocator,

        [Parameter(Mandatory=$true)]
        [string]
        # Property name
        $PropertyName,

        [Parameter(Mandatory=$true)]
        [string]
        # Property value
        $Value
    )

    $requestUrl = "$(Get-RestApiUrl $Url)/users/${UserLocator}/properties/${PropertyName}"

    return (Invoke-RestApiUsingPut $requestUrl $UserName $Password "text/plain" $Value)
}
Export-ModuleMember -Function Set-TeamCityUserProperty

<#
    .SYNOPSIS
        Set TeamCity build tags
#>
function Set-TeamCityBuildTags
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # CI server url
        $Url,

        [Parameter(Mandatory=$true)]
        [string]
        # User name
        $UserName,

        [Parameter(Mandatory=$true)]
        [string]
        # Password
        $Password,

        [Parameter(Mandatory=$true)]
        [string]
        # Build locator
        $BuildLocator,

        [Parameter(Mandatory=$true)]
        [string[]]
        # Tags
        $Tags
    )

    $requestUrl = "$(Get-RestApiUrl $Url)/builds/${BuildLocator}/tags/"
    $tagElements = -join ($Tags | % { "<tag name=`"$_`"/>" })
    $tagCount = $Tags.Length

    $value = "<?xml version=`"1.0`" encoding=`"UTF-8`" standalone=`"yes`"?><tags count=`"${tagCount}`">${tagElements}</tags>"

    return (Invoke-RestApiUsingPut $requestUrl $UserName $Password "application/xml" $value)
}
Export-ModuleMember -Function Set-TeamCityBuildTags
