﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Target Module

    .DESCRIPTION
        This file defines functions for target management
#>

$moduleFilePath = $MyInvocation.MyCommand.Path
$modulePath     = [System.IO.Path]::GetDirectoryName($moduleFilePath)
$moduleRootPath = [System.IO.Path]::GetDirectoryName($modulePath)

Import-Module "${moduleRootPath}\Path"
Import-Module "${moduleRootPath}\CiAgent"
Import-Module "${moduleROotPath}\HostBridge"

<#
    .SYNOPSIS
        Returns connected or assigned targets

    .OUTPUTS
        Array instance that contains target information objects.
#>
function Get-Targets
{
    param
    (
        [string]
        # How to connect targets
        $ConnectionType,

        [string]
        # Target Address, ignored when ConnectionType is USB. Defaults to TARGET_ADDRESS_PATTERN and then 169.* if that isn't set.
        $TargetAddressPattern = $(if ($env:TARGET_ADDRESS_PATTERN) {$env:TARGET_ADDRESS_PATTERN} else {"169.*"})
    )

    $targets = @()

    # List up agents from Target Broker
    if (Test-Path Env:\TARGET_BROKER_TARGETS) {
        -split ${env:TARGET_BROKER_TARGETS} | % { ($_ -split ",")[1] } | %{
            $targets += Get-TargetAddressFromSerial $_
        }
    }

    # List up agents from xml
    # "$_.Interface" variable used here is defined for CI only so it is not necessary to match definition in TMAPI
    if($ConnectionType -eq "Usb")
    {
        $(Get-CiAgentAssignedTargets) | ? { $_.Interface -eq "USB" } | % {
            $targets += $_.Name
        }
    } else {
        $(Get-CiAgentAssignedTargets) | ? { $_.Interface -eq "HBX" } | % {
            $serial = $_.Name
            $targets += Get-TargetAddressFromSerial $serial
        }
    }

    if(0 -eq $targets.Count)
    {
        if($ConnectionType -eq "Usb")
        {
            # Get S/N of first found SDEV/EDEV connected by USB
            Import-Module "${moduleRootPath}\Utility"
            [string[]]$list = Invoke-CriticalCommand "& `"$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe`" detect-target"

            $list = $list -match "<NoIpAddress>"
            if($list.length -eq 0){
                "No usb connected target detected."
                exit 1
            }
            $list | % {
                $targets += (($_.Split("<"))[0]).Trim()
            }
        } else {
            Import-Module "${moduleRootPath}\HostBridge"
            $targets += Get-TargetAddresses $TargetAddressPattern
        }

        Write-Host "Single Target Mode: (Target=$targets)"
    } else {
        Write-Host "Multi Target Mode:"

        # Display all targets for multi target mode
        $targets | % { Write-Host "    $_" }
    }

    return ,$targets
}

Export-ModuleMember -Function Get-Targets

function Get-PhidgetsId
{
    param
    (
        [string]
        # target name
        $TargetName
    )

    $target = $(Get-CiAgentAssignedTargets) | ? { $_.Name -eq $TargetName }
    return $target.Phidgets
}

Export-ModuleMember -Function Get-PhidgetsId

function Get-UartPort
{
    param
    (
        [string]
        # target name
        $TargetName
    )

    $target = $(Get-CiAgentAssignedTargets) | ? { $_.Name -eq $TargetName }
    return $target.Uart
}

Export-ModuleMember -Function Get-UartPort