﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        SystemTest Utility Module

    .DESCRIPTION
        This file defines functions for SystemTest operations
#>

$moduleFilePath = $MyInvocation.MyCommand.Path
$modulePath     = [System.IO.Path]::GetDirectoryName($moduleFilePath)
$moduleRootPath = [System.IO.Path]::GetDirectoryName($modulePath)

Import-Module "${moduleRootPath}\Path"

# 配列から最大値・最小値・平均値・不偏分散を取得
function Get-Statistic([array]$array)
{
    if ($array.Length -eq 0)
    {
        return
    }
    $min = $array[0]
    $max = $array[0]
    $sum = 0
    # 最大値・最小値・合計の計算
    foreach ($num in $array)
    {
        $sum += $num
        if ($num -lt $min)
        {
            $min = $num
        }
        if ($num -gt $max)
        {
            $max = $num
        }
    }
    # 平均値の計算
    $avg = $sum / $array.Length

    # 不偏分散の計算
    $sum = 0
    foreach ($num in $array)
    {
        $sum += [Math]::Pow($num - $avg,2)
    }
    $unbiasedVariance = $sum / ($array.Length - 1)

    return $max,$min,$avg,$unbiasedVariance
}
Export-ModuleMember -Function Get-Statistic

function Start-TestSuite($classname,$tests)
{
    # テスト結果出力用定義
    $xml = [xml]@'
<?xml version="1.0" encoding="UTF-8"?>
<testsuites>
  <testsuite></testsuite>
</testsuites>
'@

    $failures = 0

    $globalWatch = New-Object "System.Diagnostics.StopWatch"
    $globalWatch.Start()

    $globalWatch.Stop()

    foreach ($key in $tests.Keys)
    {
        Write-Host -n "${key}... "

        $result,$time = (& $tests[$key])

        $node = $xml.CreateElement("testcase")
        $node.SetAttribute("name", $key)
        $node.SetAttribute("status", "run")
        $node.SetAttribute("time", $time / 1000.0)
        $node.SetAttribute("classname", $classname)
        [void]$xml.testsuites.FirstChild.AppendChild($node)

        if ($result)
        {
            Write-Host "Pass"
        }
        else
        {
            Write-Host "Failed"
            $failures += 1
            $subnode = $xml.CreateElement("failure")
            $subnode.SetAttribute(
                "message",
                $tests[$key].File +
                ":" +
                $tests[$key].StartPosition.StartLine +
                " ${key} failed")
            $subnode.SetAttribute("type", "")
            [void]$node.AppendChild($subnode)
        }
    }
    # テスト結果出力
    $node = $xml.testsuites
    $node.SetAttribute("name", "AllTests")
    $node.SetAttribute("tests", $tests.Count)
    $node.SetAttribute("failures", $failures)
    $node.SetAttribute("disabled", "0")
    $node.SetAttribute("errors", "0")
    $node.SetAttribute("timestamp", (Get-Date -format s))
    $node.SetAttribute("time", $globalWatch.Elapsed.Milliseconds / 1000.0)

    $node = $node.FirstChild
    $node.SetAttribute("name", $classname)
    $node.SetAttribute("tests", $tests.Count)
    $node.SetAttribute("failures", $failures)
    $node.SetAttribute("disabled", "0")
    $node.SetAttribute("errors", "0")
    $node.SetAttribute("time", $globalWatch.Elapsed.Milliseconds / 1000.0)

    return $xml
}
Export-ModuleMember -Function Start-TestSuite
