﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Rpc Module

    .DESCRIPTION
        This file defines functions for remote procedure call(RPC)
#>

$moduleFilePath = $MyInvocation.MyCommand.Path
$modulePath     = [System.IO.Path]::GetDirectoryName($moduleFilePath)
$moduleRootPath = [System.IO.Path]::GetDirectoryName($modulePath)

Import-Module "${moduleRootPath}\Error"

# .NET の型と XML-RPC の型のマッピング
$XmlTypeMapping = @{
    [string]   = "string";
    [int]      = "int";
    [long]     = "int";
    [double]   = "double";
    [float]    = "double";
    [bool]     = "boolean";
    [DateTime] = "dateTime.iso8601";
}

<#
    .SYNOPSIS
        XML-RPC のリクエストを送信し結果を取得します。

    .OUTPUTS
        サーバからの応答を文字列で返します。

    .NOTES
        サーバからの応答がエラーの場合は停止します。
#>
function Get-XmlRpcResponse
{
    Param(
        # リクエストを送信する URL を指定します。
        [Parameter(Mandatory=$true,
                   Position=0)]
        [string]$RequestUrl,

        # メソッド名を指定します。
        [Parameter(Mandatory=$true,
                   Position=1)]
        [string]$MethodName,

        # パラメータの配列を指定します。
        # パラメータが不要な場合は省略可能です。
        [Parameter(Position=2)]
        [object[]]$Parameters=@(),

        # HttpWebRequest.GetResponseStream() のタイムアウト値(秒単位)を指定します。
        # 既定値は 100 秒 です。
        [Parameter()]
        [int]$TimeoutSec = 100
    )

    # リクエスト用の XML を生成
    $requestXml = Get-XmlRpcRequestString $MethodName $Parameters

    # HTTP でリクエストを送信する準備
    $httpRequest = [Net.HttpWebRequest]::Create($RequestUrl)
    $httpRequest.Method = "POST"
    $httpRequest.UserAgent = "Mozilla/4.0 (compatible; MSIE 6.0; Windows XP)"
    $httpRequest.ContentType = "text/xml"
    $httpRequest.Timeout = $TimeoutSec * 1000

    # コンテンツをセット
    $content = [Text.Encoding]::UTF8.GetBytes($requestXml)
    $httpRequest.ContentLength = $content.Length
    $requestStream = $httpRequest.GetRequestStream()
    $requestStream.Write($content, 0, $content.Length)
    $requestStream.Close()

    # HTTP リクエストを送信・結果を取得
    $httpResponse = $httpRequest.GetResponse()
    if($httpResponse.StatusCode -ne "OK")
    {
        # ステータスコード 200 以外はエラーとする
        Write-Error "The problem occurred while acquiring the response."
        exit 1
    }

    # 結果の XML を取得
    $responseStream = $httpResponse.GetResponseStream()
    $responseXml = ((New-Object IO.StreamReader($responseStream)).ReadToEnd())
    $responseStream.Close()

    return $responseXml
}

# XML-RPC のリクエスト用の XML 文字列を取得する
function Get-XmlRpcRequestString([string]$method, [object[]]$parameters)
{
    $xml = New-Object Xml

    # xmlDeclaration
    $xmlDeclaration = $xml.CreateXmlDeclaration("1.0", "UTF-8", $null)

    # methodName
    $methodName = $xml.CreateElement("methodName")
    $methodName.InnerText = $method

    # params
    $params = $xml.CreateElement("params")
    Append-XmlRcpParamElements $xml $params $parameters

    # methodCall
    $methodCall = $xml.CreateElement("methodCall")
    [void]$methodCall.AppendChild($methodName)
    [void]$methodCall.AppendChild($params)

    # xml
    [void]$xml.AppendChild($xmlDeclaration)
    [void]$xml.AppendChild($methodCall)

    return $xml.OuterXml
}

# Param 要素を追加する
function Append-XmlRcpParamElements([Xml]$xml, [Xml.XmlElement]$parent, [object[]]$parameters)
{
    # 空の配列を準備
    $params = @()

    # param 要素を準備
    for($i = 0; $i -lt $parameters.Length; $i++)
    {
        # param
        $param = $xml.CreateElement("param")
        Append-XmlRcpValueElement $xml $param $parameters[$i]

        # 親に追加
        [void]$parent.AppendChild($param)
    }
}

# Param の内容を追加する
function Append-XmlRcpValueElement([xml]$xml, [Xml.XmlElement]$parent, [object]$parameter)
{
    # value
    $value = $xml.CreateElement("value")

    if($parameter -is [HashTable])
    {
        # Hashtable の場合
        Append-XmlRcpStructElement $xml $value $parameter
    }
    elseif($parameter -is [Array])
    {
        # 配列の場合
        Append-XmlRpcDataElement $xml $value $parameter
    }
    else
    {
        $type = $XmlTypeMapping[$parameter.GetType()]
        $typeElement = $xml.CreateElement("${type}")

        # それ以外の場合
        if($parameter -is [DateTime])
        {
            $typeElement.InnerText = $parameter.ToString("yyyyMMddTHH:mm:ss")
        }
        else
        {
            $typeElement.InnerText = "${parameter}"
        }

        [void]$value.AppendChild($typeElement)
    }

        # 親に追加
    [void]$parent.AppendChild($value)
}

# Struct 要素を追加する
function Append-XmlRcpStructElement([xml]$xml, [Xml.XmlElement]$parent, [HashTable]$parameter)
{
    # struct
    $struct = $xml.CreateElement("struct")

    foreach ($param in $parameter.GetEnumerator())
    {
        # member
        $member = $xml.CreateElement("member")

        # name
        $name = $xml.CreateElement("name")
        $name.InnerText = $param.Key

        [void]$member.AppendChild($name)

        Append-XmlRcpValueElement $xml $member $param.Value

        [void]$struct.AppendChild($member)
    }

    # 親に追加
    [void]$parent.AppendChild($struct)
}

# Data 要素を追加する
function Append-XmlRpcDataElement([xml]$xml, [Xml.XmlElement]$parent, [Array]$parameter)
{
    # data
    $data = $xml.CreateElement("data")

    for($i = 0; $i -lt $parameter.Length; $i++)
    {
        Append-XmlRcpValueElement $xml $data $parameter[$i]
    }

    # array
    $array = $xml.CreateElement("array")
    [void]$array.AppendChild($data)

    # 親に追加
    [void]$parent.AppendChild($array)
}


Export-ModuleMember Get-XmlRpcResponse
