﻿<#
    Copyright (C)Nintendo. All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.
 
    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
    Zarf定義のDescription を集めて、XMLファイルとして書き出します。

    .DESCRIPTION
    Zarf定義のDescription を集めて、XMLファイルとして書き出します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
[CmdletBinding()]
Param(
    # 出力XMLファイルのパスを指定します。
    [parameter(Mandatory = $true,
               Position = 0)]
    [string]$Path,

    # Zarf定義ファイルが置かれているディレクトリを指定します。
    [parameter()]
    [string]$ZarfDir = "Integrate/Packages/ZarfDefinitions",

    # Zarf定義ファイルが置かれているディレクトリを指定します。
    [parameter()]
    [ValidateSet("Ja", "En")]
    [string]$Lang = 'Ja'
)

function Import-Yaml
{
    Param(
        [parameter(Mandatory = $true)]
        [string]$Path
    )

    $deserializer = New-Object YamlDotNet.Serialization.Deserializer ($null, $null)
    try {
        $sr = New-Object IO.StreamReader ($Path, [Text.Encoding]::UTF8, $true)
        $deserializer.Deserialize($sr)
    } finally {
        if ($sr) {
            $sr.Close()
        }
    }
}

function Export-MsgXml
{
    Param(
        [parameter(Mandatory = $true)]
        [string]$Path,

        [parameter(Mandatory = $true)]
        [AllowEmptyString()]
        [AllowEmptyCollection()]
        $lines
    )

    $xmlDoc = New-Object System.Xml.XmlDocument

    # Create an XML declaration. 
    $xmlDoc.AppendChild($xmlDoc.CreateXmlDeclaration("1.0", "UTF-8", $null)) > $null

    $rootEle = $xmlDoc.AppendChild($xmlDoc.CreateElement("Descriptions"))

    for ($i = 0; $i -lt $lines.Length; $i++) {

        $descEle = $rootEle.AppendChild($xmlDoc.CreateElement("Description"))
        $textEle = $descEle.AppendChild($xmlDoc.CreateElement("Text"))

        $descEle.SetAttribute("name", $lines[$i].Name)
        $line = $lines[$i].Desc -replace '\n',"`r`n"
        if ($line) {
            $textEle.AppendChild($xmlDoc.CreateTextNode($line)) > $null
        }
    }

    $xmlDoc.Save([IO.Path]::Combine($PWD, $Path))
}

$sigloRoot = (Join-Path $MyInvocation.MyCommand.Path ../../../.. | Resolve-Path).Path

Add-type -Path $sigloRoot/Externals/Binaries/YamlDotNet-3.0.0/YamlDotNet.dll

$textsJa =
    Get-ChildItem $sigloRoot/$ZarfDir -Include *.zarf.yml,*.zarf.yml.template -Recurse |
    Where-Object { $_.Extension -ne '.yml' -or !(Test-Path ($_.FullName + ".template")) } |
    ForEach-Object {
        $zarfDef = Import-Yaml $_.FullName
        $descJa = $zarfDef["Zarf"]["Description"][$Lang]
        if ($descJa) {
            @{
                Desc = $descJa
                Name = $zarfDef["Zarf"]["Name"]
            }
        }

        $postInstalls = $zarfDef['Zarf']['Instructions']['PostInstall']
        if ($postInstalls) {
            foreach ($cmd in $postInstalls) {
                $handler = $cmd['Handler']
                if ($handler -like 'Register-Dashboard-*') {
                    $dashboardDetails = $cmd['DashboardDetails']
                    $displayNameJa = $dashboardDetails['GroupDisplayName'][$Lang]
                    if ($displayNameJa) {
                        @{
                            Desc = $displayNameJa
                            Name = $zarfDef["Zarf"]["Name"] + ':' + $dashboardDetails['GroupName'] + ':GroupDisplayName'
                        }
                    }

                    $displayTextJa = $dashboardDetails['DisplayText'][$Lang]
                    if ($displayTextJa) {
                        @{
                            Desc = $displayTextJa
                            Name = $zarfDef["Zarf"]["Name"] + ':' + $dashboardDetails['GroupName'] + ':DisplayText'
                        }
                    }
                }
            }
        }
    }

Export-MsgXml $Path $textsJa

