<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Test on Target

    .DESCRIPTION
        Run all executables in specified directory as test.
#>

param
(
    [string]
    # Target Address Pattern
    $AddressPattern = $env:TARGET_ADDRESS_PATTERN,

    [string]
    # The platform name
    $Platform,

    [int]
    # The number of seconds before it times out.
    $Timeout = 300,

    [string]
    # The build type
    $BuildType = "Develop",

    [string]
    # The root of test directory
    $Path = "Tests\Outputs",

    [string]
    # Exclusion files separated with a comma
    $ExcludeNames = ""
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\TargetManager"
Import-Module "${scriptDirectoryPath}\Modules\Runner"
Import-Module "${scriptDirectoryPath}\Modules\HostBridge"
Import-Module "${scriptDirectoryPath}\Modules\Utility"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

$logDir = [System.IO.Path]::GetFileNameWithoutExtension($scriptPath)
$logDir = "${scriptDirectoryPath}\${logDir}"
$logDir = $logDir.Replace((Get-NintendoSdkIntegrateRootPath), "")
$logDir = (Get-NintendoSdkIntegrateOutputRootPath) + $logDir
$logDir = $logDir + "\" + (Get-Date -Format "yyyyMMddHHmmss")
[void](New-Item $logDir -ItemType Directory -ErrorAction "Stop")

# collect executable file path
$targetList = [System.IO.Path]::Combine((Get-NintendoSdkRootPath), ${Path}) | Get-ChildItem -Include "*.kip","*.nsp" -Recurse | %{ $_.FullName} | Where-Object { $_.Contains("\${BuildType}\") }

if ($ExcludeNames)
{
    $excludeLists = $ExcludeNames -csplit "\s*,\s*"

    # Remove exclude tests
    $excludeLists | %{
        $targetList -match $_ | %{ $targetList[[Array]::IndexOf($targetList, $_)] = $null}
    }
}

if (-not $AddressPattern)
{
    # If AddressPattern is not specified and env:TARGET_ADDRESS_PATTERN is not difined
    $AddressPattern = "169.*";
}

$TestResult = 1

try
{
    $targetList | % {

        $logFile = $logDir + "\" + [System.IO.Path]::GetFileNameWithoutExtension($_) + ".log"
        Write-Host "Start ${_}"
        Write-Host "logFile: ${logFile}"

        # Get the address of the HostBridge
        $TargetAddress = Get-SigleTargetAddress -AddressPattern $AddressPattern

        # テストを実行
        Invoke-Expression "& `"$(Get-NintendoSdkRootPath)\Tools\CommandLineTools\RunOnTargetPrivate.exe`" run $_ --pattern-success-exit `"\[  PASSED  \] .+ tests?.`" --pattern-failure-exit `"\[  FAILED  \]`" --pattern-failure-exit `"Assertion Failure:`" --failure-timeout $Timeout --verbose --monitor-serial --hostbridge $TargetAddress | Set-Content -Passthru $logFile | Write-Host"
        $TestResult = $LastExitCode

        # TORIAEZU: 環境変数を見てリセットする
        if ($TestResult -ne 0 -or $env:REQUIRE_TARGET_RESET -eq "true")
        {
            $ControlTarget = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe"
            Invoke-CriticalCommand "& `"$ControlTarget`" reset"
        }
    }
}
finally
{
}

# Pass exitCode to TestRunner's result
exit $TestResult
