﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke a PowerShell script

    .DESCRIPTION
        Run a PowerShell script with configuration files
#>


$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Yaml"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

function Test-OptionNode([object]$node)
{
    if ($node -isnot [System.Xml.XmlElement])
    {
        return $false
    }

    if ($node.Name -isnot [string])
    {
        return $false
    }

    if ($null -ne $node.Args)
    {
        if ($node.Args -isnot [System.Xml.XmlElement])
        {
            return $false
        }

        foreach ($childNode in $node.Args.ChildNodes)
        {
            if ($childNode -isnot [System.Xml.XmlElement])
            {
                return $false
            }

            if (1 -ne $childNode.ChildNodes.Count)
            {
                return $false
            }

            if ($childNode.FirstChild -isnot [System.Xml.XmlText])
            {
                return $false
            }
        }
    }

    return $true
}

function Get-StringOfOptionNode([System.Xml.XmlElement]$node)
{
    $arguments = New-Object "System.Collections.Generic.List[string]"
    $argumentFormat = "Default"

    if ($null -ne $node.Args)
    {
        $node.Args.ChildNodes | % { $arguments.Add($_.InnerText) }
    }
    if ($null -ne $node.ArgFormat)
    {
        $argumentFormat = $node.ArgFormat
    }

    if (0 -eq $arguments.Count)
    {
        return $node.Name
    }
    else
    {
        if ("EqualCommaSeparatedValues" -eq $argumentFormat)
        {
            # name="value1,value2,..." の形式で返す
            return "$($node.Name)=`"$($arguments -join `",`")`""
        }
        elseif ("CommaSeparatedValues" -eq $argumentFormat)
        {
            # name value1,value2,... の形式で返す
            return "$($node.Name) $($arguments -join `",`")"
        }        
        elseif ("Default" -eq $argumentFormat `
                -or "RepeatOptionForEachArgument" -eq $argumentFormat)
        {
            # name value1 name value2 ... を引数の数だけ繰り返す
            $ret = @()
            $arguments | % {
                $ret = $ret + @($node.Name, $_)
            }
            return ,$ret
        }
        else
        {
            throw "Unknown ArgFormat '${argumentFormat}'"
        }
    }
}

if ($args.Length -lt 1)
{
    throw "Script name is required"
}

if ($args.Length -lt 2)
{
    throw "Configuration names are required"
}

$scriptName  = $args[0]

$configNames = $args[1]

$arguments = New-Object "System.Collections.Generic.List[object]"
for ($i = 2; $i -lt $args.Length; ++$i)
{
    $arguments.Add($args[$i])
}

$scriptFile = "${scriptDirectoryPath}\${scriptName}.ps1"
if (-not (Test-Path $scriptFile))
{
    Write-Warning "'${scriptName}' is not available"
    exit 0
}

$rootPath = Get-NintendoSdkIntegrateRootPath

$configFiles = New-Object "System.Collections.Generic.List[object]"
$configNames | % {
    if ("None" -ceq $_) { return }

    $configFile = "${rootPath}\Resources\Scripts\Configs\${scriptName}\${_}.yml"

    if (-not (Test-Path $configFile))
    {
        Write-Warning "'${_}' is not available"
        exit 0
    }

    $configFiles.Add($configFile)
}

$options = New-Object "System.Collections.Generic.List[string]"

$configFiles | % {
    $configName = [System.IO.Path]::GetFileName($_)

    $rawData = (Get-Content $_) -join [System.Environment]::NewLine
    if ([string]::IsNullOrEmpty($rawData))
    {
        throw "'${configName}' is empty"
    }

    (ConvertFrom-Yaml $rawData).DocumentElement.ChildNodes | % {
        if (-not (Test-OptionNode $_))
        {
            throw "'${configName}' is not a valid configuration file"
        }

        (Get-StringOfOptionNode $_) | % { $options.Add($_) }
    }
}


try {
    if ($env:TEAMCITY_VERSION) {
      Write-Host "##teamcity[blockOpened name='$scriptName $configName']"
    }
    Invoke-Expression "& `"$scriptFile`" $($options -join ' ') $($arguments -join ' ')"
}
finally {
    if ($env:TEAMCITY_VERSION) {
      Write-Host "##teamcity[blockClosed name='$scriptName $configName']"
    }
}

exit $LastExitCode
