﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Safemode Test on Target

    .DESCRIPTION
        Boot Safemode.
#>

param
(
    [string]
    # Target Address Pattern
    $AddressPattern = $env:TARGET_ADDRESS_PATTERN,

    [string]
    # The platform name
    $Platform = "NX64",

    [int]
    # The number of seconds before it times out.
    $Timeout = 90,

    [string]
    # The build type
    $BuildType = "Develop",

    [string]
    $TargetName,

    [string]
    $TargetInterface = "Ethernet",

    [string]
    $TargetAddress
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\HostBridge"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Utility"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if ([string]::IsNullOrEmpty($AddressPattern))
{
    # If AddressPattern is not specified and env:TARGET_ADDRESS_PATTERN is not
    # difined
    $AddressPattern = "^169\.*"
}

$VsPlatform = switch ($Platform)
{
    "NX32"      { "NX-NXFP2-a32" }
    "NXFP2-a32" { "NX-NXFP2-a32" }
    "NX64"      { "NX-NXFP2-a64" }
    "NXFP2-a64" { "NX-NXFP2-a64" }
    default     { $_ }
}

if ([string]::IsNullOrEmpty($TargetName))
{
    if ("Ethernet" -cne $TargetInterface)
    {
        throw "TargetName not specified"
    }
    else
    {
        $TargetName = Get-SigleTargetAddress -AddressPattern $AddressPattern

        $TargetAddress = $TargetName
    }
}

if ([string]::IsNullOrEmpty($TargetAddress))
{
    if ("Ethernet" -ceq $TargetInterface)
    {
        throw "TargetAddress not specified"
    }
}

# TORIAEZU :
#  dll が読めないので、暫定対応として、Outputs 以下の ControlTarget を実行
$ControlTarget = Get-NintendoSdkRootPath
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Programs\Chris\Outputs")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "x86")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Tools\RunnerTools")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "ControlTargetPrivate")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Release")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "ControlTargetPrivate.exe")

$ReadHbLog = Get-NintendoSdkRootPath
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "Tools")
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "CommandLineTools")
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "ReadHostBridgeLog.exe")

$logDir = $scriptDirectoryPath
$logDir = $logDir.Replace(
    (Get-NintendoSdkIntegrateRootPath),
    (Get-NintendoSdkIntegrateOutputRootPath))
$logDir = [IO.Path]::Combine($logDir, [IO.Path]::GetFileNameWithoutExtension($scriptPath))
$logDir = [IO.Path]::Combine($logDir, ((Get-Date -Format "yyyyMMddHHmmss") + "-" + $TargetName))
[void](New-Item $logDir -ItemType Directory -ErrorAction "Stop")

$logFile = [IO.Path]::Combine($logDir, ("SafemodeTest.log"))

Write-Host "logFile: ${logFile}"

$TestResult = 1
$BackgroundJobResult = 1

if ("Ethernet" -ieq $TargetInterface)
{
    # セーフモードに入る
    $list = ($ControlTarget, $TargetName)
    $job = Start-Job -ArgumentList $list -ScriptBlock {
        param($path, $name)
        Write-Host "Sub: ${path}, ${name}"
        $subCommand  = ("&", "`"$path`"", "boot-safemode")
        $subCommand += ("--target", $name)
        $subCommand = $subCommand -join " "
        Invoke-Expression $subCommand
    }

    # HostBridge Log の監視
    $command  = ("&", "`"$ReadHbLog`"")
    $command += ("--address", $TargetAddress)
    $command += ("--verbose", "--failure-timeout", "$Timeout")
    $command += ("--pattern-success-exit", "`"\[SafeMode\] Network update done.`"")   # 更新まで完了
    $command += ("--pattern-success-exit", "`"\[SafeMode\] network available = 0`"")  # セーフモード起動確認のみ
    $command += ("--pattern-failure-exit", "`"\[  FAILED  \]`"")
    $command += ("--pattern-failure-exit", "`"\[TestOnTarget\] FAILED`"")
    $command += ("--pattern-failure-exit", "`"Assertion Failure:`"")
    $command += ("--pattern-failure-exit", "`"Break\(\) called`"")
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host

    $ReadHostBridgeLogResult = $LastExitCode

    # Background Job の完了待ちと結果のチェック
    Wait-Job -id $job.Id
    if($job.State -eq "Completed")
    {
        $BackgroundJobResult = 0
    }

    Write-Output "ReadHostBridgeLogResult = $ReadHostBridgeLogResult"
    Write-Output "BackgroundJobResult     = $BackgroundJobResult"

    $TestResult = $ReadHostBridgeLogResult + $BackgroundJobResult
}
else
{
    Write-Warning "$TargetInterface is not supported"
}

if ((0 -ne $TestResult) -or ("true" -eq $env:REQUIRE_TARGET_RESET))
{
    # TargetManager を落とさないとリセットできないため、一度落とす
    Invoke-Expression "& `"${scriptDirectoryPath}\Stop-TargetManager.ps1`""

    Invoke-CriticalCommand "& `"$ControlTarget`" reset --target $TargetName"
}

# Pass exitCode to TestRunner's result
Write-Output "TestResult              = $TestResult"
exit $TestResult
