﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke MSBuild

    .DESCRIPTION
        Run MSBuild for solution files
#>

param
(
    [string[]]
    # Paths to the solution files
    $File,

    [string]
    # Path to the root directory of solution files
    $Path,

    [string[]]
    # Specify targets
    $Target = (,"Build"),

    [string[]]
    # Qualify platforms
    $Platform,

    [string[]]
    # Qualify configurations
    $Configuration,

    [string[]]
    # Qualify specs
    $Spec,

    [string[]]
    # Qualify Visual Studio Verson
    $VSVersion,

    [string[]]
    # Exclusion files separated with a comma
    $ExcludeNames,

    [switch]
    # Enable the CI mode
    $CIMode
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\TeamCity"
Import-Module "${scriptDirectoryPath}\Modules\WindowsErrorReporting"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

$wer = Get-WindowsErrorReporting

if ( -not (($null -ne $File) -xor $Path) )
{
    Write-ErrorMessage "'Path' and 'File' are either-or options"
    exit 1
}

if ( $Path -and (-not (Test-Path $Path -Type Container)) )
{
    Write-ErrorMessage "Directory '${Path}' not found"
    exit 1
}

if ($null -ne $File)
{
    $File | % {
        if ( -not (Test-Path $_ -Type Leaf) )
        {
            Write-ErrorMessage "File '$_' not found"
            exit 1
        }
    }
}

$solutions = if ($null -ne $File)
{
    ,($File | % { Convert-Path $_ })
}
else
{
    ,[System.IO.Directory]::GetFiles(
        (Convert-Path $Path),
        "*.sln",
        [System.IO.SearchOption]::AllDirectories)
}

$assemblyName = New-Object "System.Collections.Generic.List[string]"
$assemblyName.Add("Microsoft.Build.Engine")
$assemblyName.Add("Version=3.5.0.0")
$assemblyName.Add("Culture=neutral")
$assemblyName.Add("PublicKeyToken=b03f5f7f11d50a3a")
[void][System.Reflection.Assembly]::Load($assemblyName -join ", ")

$binPath = [System.IO.Path]::GetDirectoryName((Get-MSBuildX86Path))
$engine  = New-Object "Microsoft.Build.BuildEngine.Engine" $binPath

$errorSlns = New-Object "System.Collections.Generic.List[string]"

$solutions | ? { $ExcludeNames -notcontains [System.IO.Path]::GetFileName($_) } | % {
    $solution = $_

    Write-Host "loading ${solution}..."
    $project = New-Object "Microsoft.Build.BuildEngine.Project" ($engine, "4.0")

    if ( $null -ne $Spec )
    {
        $matched = $false
        $Spec | ? { $solution -cmatch $_ } | % { $matched = $true }
       
        if( $false -eq $matched ) 
        {
            Write-Host "Skipped."
            return
        }
    }

    if ( $null -ne $VSVersion )
    {
        $matched = $false
        $VSVersion | ? { $solution -cmatch $_ } | % { $matched = $true }
       
        if( $false -eq $matched ) 
        {
            Write-Host "Skipped."
            return
        }
    }

    try
    {
        $testName = "ProjectFileVerificationTest"
        if ( $CIMode ) { Enter-TestingSessionForTeamCity $solution $testName }
        $project.Load($solution)
        if ( $CIMode ) { Exit-TestingSessionForTeamCity $solution $testName }
    }
    catch
    {
        $message = $_.Exception.Message
        if ( $CIMode )
        {
            Exit-TestingSessionForTeamCity $solution $testName $message
        }
        else
        {
            $Host.UI.WriteErrorLine("Error: " + $message)
        }
        $errorSlns.Add($solution)
        return
    }

    $cfgSet = New-Object "System.Collections.Generic.Dictionary[string,int]"
    $project.PropertyGroups | % {
        $_ | % {
            if ( "CurrentSolutionConfigurationContents" -eq $_.Name )
            {
                $slnConfig = ([xml]$_.Value)["SolutionConfiguration"]
                $prgConfig = $slnConfig.FirstChild.FirstChild.Value
                $cfgSet[$prgConfig] = 0
            }
        }
    }

    $cfgSet.Keys | % {
        $cfg, $plt = $_.Split("|", 2)
        $rawCfg = $cfg

        $plt = if ( $plt -eq "AnyCPU" ) { "Any CPU" } else { $plt }

        $cfg = if ( $cfg -match "Debug") { "Debug" } else { $cfg }
        $cfg = if ( $cfg -match "Develop") { "Develop" } else { $cfg }
        $cfg = if ( $cfg -match "Release") { "Release" } else { $cfg }

        if (($null -ne $Platform) -and ($Platform -cnotcontains $plt))
        {
            return
        }

        if (($null -ne $Configuration) -and ($Configuration -cnotcontains $rawCfg))
        {
            return
        }

        Write-Host "building for Platform:${plt}, Configuration:${rawCfg}..."
        $command = New-Object "System.Collections.Generic.List[string]"
        $command.Add("`"$(Get-MSBuildX86Path)`"")
        $command.Add("`"${solution}`"")
        $command.Add("/m")
        $command.Add("/nologo")
        $command.Add("/v:m")
        $command.Add("/t:`"" + ($Target -join ";") + "`"")
        $command.Add("/p:Platform=`"${plt}`"")
        $command.Add("/p:Configuration=`"${cfg}`"")
        $testName = "BuildVerificationTest(${cfg}|${plt})"
        if ( $CIMode ) { Enter-TestingSessionForTeamCity $solution $testName }
        try
        {
            Disable-WindowsErrorReporting
            Invoke-Expression "& $($command -join ' ')"
        }
        finally
        {
            if ($wer -ceq "enabled")
            {
                Enable-WindowsErrorReporting
            }
        }
        if ( $LastExitCode -ne 0 )
        {
            if ( $CIMode )
            {
                $message = "MSBuild.exe returned exit code " + $LastExitCode
                Exit-TestingSessionForTeamCity $solution $testName $message
            }
            $errorSlns.Add($solution + "(${cfg}|${plt})")
            return
        }
        if ( $CIMode ) { Exit-TestingSessionForTeamCity $solution $testName }
    }
}

Write-Host ""
if ( $errorSlns.Count -eq 0 )
{
    Write-Host "done."
    exit 0
}
else
{
    $builder = New-Object "System.Text.StringBuilder"
    [void]$builder.AppendLine("failed to build the following solution files")
    $errorSlns | % { [void]$builder.AppendLine("- " + $_) }
    Write-ErrorMessage $builder.ToString().TrimEnd()
    if ( $CIMode ) { exit 0 } else { exit 1 }
}
