﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Get the directory paths of the tests that need to be run

    .DESCRIPTION
        List the modified tests accoding to the result file
#>

param
(
   [string]
    # Path to the result file
    $Path
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

$contexts = (
    @{ RootPath = "Integrate";              SourceDirectory = "Sources" },
    @{ RootPath = "Programs\Alice";         SourceDirectory = "Sources" },
    @{ RootPath = "Programs\Chris";         SourceDirectory = "Sources" },
    @{ RootPath = "Programs\Eris";          SourceDirectory = "Sources" },
    @{ RootPath = "Programs\Iris";          SourceDirectory = "Sources" },
    @{ RootPath = "Programs\NintendoWare";  SourceDirectory = "Sources" },
    @{ RootPath = "Tests";                  SourceDirectory = "" }
)

$nintendoSdkRootPath = Get-NintendoSdkRootPath

$excludeDirectoryNames = (
    "Configs",
    "Definitions",
    "Include",
    "Intermediates",
    "Libraries",
    "Outputs",
    "Processes",
    "Properties",
    "Resources",
    "SystemData"
)

$projectFileExtensions = (
    ".csproj",
    ".vcxproj"
)

if ([string]::IsNullOrEmpty($Path))
{
    throw "Path is not specified"
}

if (-not (Test-Path $Path -PathType Leaf))
{
    throw "File '$Path' not found"
}

$contexts | % {
    $context = $_
    $context.SearchPath = [IO.Path]::Combine(
        [IO.Path]::Combine($nintendoSdkRootPath, $context.RootPath), $context.SourceDirectory)
    $context.YmlPath = ""
    $context.ModuleNames = @{}
    $context.TargetPaths = @{}
}

$contexts | % {
    $context = $_

    $rootPath = [Text.RegularExpressions.Regex]::Escape($context.RootPath)

    $rootPathRegex = New-Object "Text.RegularExpressions.Regex" (
        "\\${rootPath}\\Outputs\\(.+)",
        [Text.RegularExpressions.RegexOptions]::Compiled)

    $moduleNameRegex = New-Object "Text.RegularExpressions.Regex" (
        "([^\\]+)\\(Debug|Develop|Release)\\",
        [Text.RegularExpressions.RegexOptions]::Compiled)

    foreach ($i in Get-Content $Path)
    {
        $match = $rootPathRegex.Match($i)

        if ($match.Success)
        {
            $match = $moduleNameRegex.Match($match.Groups[1].Value)

            if ($match.Success)
            {
                $context.ModuleNames[$match.Groups[1].Value] = $true
            }
        }
    }
}

$regexs = @{}

$regexs.ModuleName = New-Object "Text.RegularExpressions.Regex" (
    "(.+)-spec",
    [Text.RegularExpressions.RegexOptions]::Compiled)

$regexs.SubRootPath = New-Object "Text.RegularExpressions.Regex" (
    "(.+\\Sources\\Tests)(\\Tools)?\\[^\\]+",
    [Text.RegularExpressions.RegexOptions]::Compiled)

$regexs.TestSubRootPath = New-Object "Text.RegularExpressions.Regex" (
    "\\Tests\\(.+)\\Sources\\",
    [Text.RegularExpressions.RegexOptions]::Compiled)

function Get-TargetTest([hashtable]$context, [hashtable]$regexs)
{
    $backup = @{ SearchPath = $context.SearchPath; YmlPath = $context.YmlPath }

    try
    {
        foreach ($i in Get-ChildItem -Path $context.SearchPath)
        {
            if ("Directory" -cne $i.Attributes)
            {
                if ($i.Name.EndsWith(".testlist.yml"))
                {
                    $context.YmlPath = $i.DirectoryName
                }
            }
            elseif ("Properties" -ceq $i.Name)
            {
                foreach ($j in Get-ChildItem -Path $i.FullName)
                {
                    if ("Directory" -cne $j.Attributes)
                    {
                        if ($j.Name.EndsWith(".testlist.yml"))
                        {
                            $context.YmlPath = $i.Parent.FullName
                        }
                    }
                }
            }
        }

        foreach ($i in Get-ChildItem -Path $context.SearchPath)
        {
            if ("Directory" -ceq $i.Attributes)
            {
                if ($excludeDirectoryNames -notcontains $i.Name)
                {
                    $context.SearchPath = $i.FullName

                    Get-TargetTest $context $regexs
                }
            }
            elseif ($projectFileExtensions -contains $i.Extension)
            {
                if (-not [string]::IsNullOrEmpty($context.YmlPath))
                {
                    $name = [IO.Path]::GetFileNameWithoutExtension($i.Name)

                    $match = $regexs.ModuleName.Match($name)

                    if ($match.Success)
                    {
                        $name = $match.Groups[1].Value
                    }

                    if ($context.ModuleNames[$name])
                    {
                        $targetPath = $context.YmlPath

                        $match = $regexs.SubRootPath.Match($targetPath)

                        if ($match.Success)
                        {
                            $targetPath = $match.Groups[0].Value

                            $targetSubPath = $match.Groups[1].Value

                            $match = $regexs.TestSubRootPath.Match($targetPath)

                            if ($match.Success)
                            {
                                if ("Tools" -cne $match.Groups[1].Value)
                                {
                                    $targetPath = $targetSubPath
                                }
                            }
                        }

                        $context.TargetPaths[$targetPath] = $true
                    }
                }
            }
        }
    }
    finally
    {
        $backup.Keys | % { $context[$_] = $backup[$_] }
    }
}

$contexts | ? { 0 -lt $_.ModuleNames.Count } | % { Get-TargetTest $_ $regexs }

$contexts | % { $_.TargetPaths.Keys } | Sort-Object
