﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Copy ListedFiles

    .DESCRIPTION
        Copy Listed Files to Output Directory
#>

Param(
  # ファイルの一覧
  [string]$List,

  # コピー先ディレクトリ
  [string]$OutputDirectory,
  
  # zip を作成するかどうか
  [switch]$Archive
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
$projectRootPath     = Join-Path $scriptDirectoryPath "..\\..\\"

Import-Module "${ScriptDirectoryPath}\Modules\Error"
Import-Module "${ScriptDirectoryPath}\Modules\MakeLink"

# 成果物のファイルパスから、格納先のアーカイブ名を求めます。
function Get-TargetArchiveName( $filepath )
{
    # ライブラリ（ビルドターゲットごとに別のアーカイブとする）
    if ($filepath -match "^Programs\\(.*?)\\Libraries\\(.*?)\\.*$")
    {
        return "PrebuiltResults.$($matches[1]).Libraries.$($matches[2])"
    }
    # プロセス（ビルドターゲットごとに別のアーカイブとする）
    if ($filepath -match "^Programs\\(.*?)\\Outputs\\(.*?)\\Processes\\.*$")
    {
        return "PrebuiltResults.$($matches[1]).Processes.$($matches[2])"
    }
    # システムイメージ（ビルドターゲットごとに別のアーカイブとする）
    if ($filepath -match "^Programs\\(.*?)\\Outputs\\(.*?)\\SystemImages\\.*$")
    {
        return "PrebuiltResults.$($matches[1]).SystemImages.$($matches[2])"
    }
    # ヘッダ
    if ($filepath -match "^Programs\\.*?\\Outputs\\Include\\.*$")
    {
        return "PrebuiltResults.Headers"
    }
    # ツール
    if ($filepath -match "^Tools\\.*$")
    {
        return "PrebuiltResults.Tools"
    }
    
    # その他
    throw "ターゲットアーカイブが不明な成果物です。: $filepath"
}

# 指定のディレクトリから、アーカイブディレクトリだけをリストアップします。
function List-TargetArchiveDirectories( $searchDirectory )
{
	return Get-ChildItem -Path $searchDirectory -Filter "PrebuiltResults*" | ?{ $_.PSIsContainer } | Select-Object FullName;
}

# 例外ハンドラ
trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if ([string]::IsNullOrEmpty($List))
{
    throw "List が指定されていません"
}

if ([string]::IsNullOrEmpty($OutputDirectory))
{
    throw "OutputDirectory が指定されていません"
}

# 各アーカイブが抱える成果物のリストを置くディレクトリ
$prebuiltResultsListDirectory = Join-Path $OutputDirectory "PrebuiltResultsList"

if (-not (Test-Path $prebuiltResultsListDirectory -PathType Container))
{
    New-Item $prebuiltResultsListDirectory -Type Directory -Force -ErrorAction Stop | Out-Null
}
    
# ターゲットアーカイブごとにリストとディレクトリを作成し、成果物へのハードリンクを作成します。
Write-Output "Collect results and make lists"
foreach ($file in Get-Content $List -ErrorAction Stop)
{
    $source = $file
    $targetArchiveName = Get-TargetArchiveName $source
    $targetArchiveListPath = Join-Path $prebuiltResultsListDirectory "$targetArchiveName.list"
    $destination = Join-Path $OutputDirectory $targetArchiveName | Join-Path -child $file
    $destinationParentDirectory = Split-Path $destination -Parent

    if (-not (Test-Path $destinationParentDirectory -PathType Container))
    {
        New-Item $destinationParentDirectory -Type Directory -Force -ErrorAction Stop | Out-Null
    }
    
    New-HardLink -NewLink $destination -Source $source
    Add-Content $targetArchiveListPath -Value $source -Encoding UTF8
}

# 各ディレクトリを圧縮します。
if ($Archive)
{
    Write-Output "Make archives"
   
    $zipCommand = Join-Path $projectRootPath "Externals\\Binaries\\7-Zip\\7z.exe"

    foreach ($directory in List-TargetArchiveDirectories $OutputDirectory)
    {
        $directoryPath = $directory.FullName
        $archivePath = "$directoryPath.zip";
        
        Write-Output "$directoryPath => $archivePath"

        # SIGLONTD-9997: Start-Process は、実行時間の短いコマンドに対して失敗を返すことがある問題がある。使用しないようにする。
        # アーカイブに格納されるパスが相対パスになるようにするため、カレントディレクトリを $directoryPath にして 7z を実行する。
        Push-Location $directoryPath
        & $zipCommand a $archivePath .
        Pop-Location
    }
}
else
{
    "Make archives...Skip"
}
