﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        TeamCity のユーザの VCS 設定をチェックし、必要に応じて設定を変更します。

    .DESCRIPTION
        TeamCity の全てのユーザの VCS 設定がメールアカウントと一致しているかどうかをチェックし、
        一致していない場合は VCS 設定にメールアカウントを設定します。
        メールアドレスが設定されていないユーザを見つけた場合は警告を出します。
#>

Param(
  # 認証情報が書かれた設定ファイルを指定します
  # 以下、設定ファイルのサンプルです
  # <?xml version="1.0"?>
  # <Configuration>
  #   <User>ユーザ名</User>
  #   <Password>パスワード</Password>
  # </Configuration>
  [string]$AuthenticationFile
)

$ScriptPath = $MyInvocation.MyCommand.Path
$ScriptName = [System.IO.Path]::GetFileNameWithoutExtension($ScriptPath)
$ScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($ScriptPath)

Import-Module "${ScriptDirectoryPath}\Modules\Error"
Import-Module "${ScriptDirectoryPath}\Modules\Path"
Import-Module "${ScriptDirectoryPath}\Modules\TeamCity"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if ([string]::IsNullOrEmpty($AuthenticationFile))
{
    throw "AuthenticationFile が指定されていません"
}

# 設定ファイルの読み込み
[Xml]$Config = Get-Content $AuthenticationFile
$User = $Config.Configuration.User
$Password = $Config.Configuration.Password

# TeamCity
$TeamCityUrl = "http://devsedci01.ncl.nintendo.co.jp/TeamCity"


# ユーザ一覧の取得
$Users = Get-TeamCityUsers $TeamCityUrl $User $Password


# それぞれのユーザに対して設定を確認・反映
foreach ($TeamCityUser in $Users.users.user)
{
    # IDとユーザ名を取得
    $UserId = $TeamCityUser.id
    $UserName = $TeamCityUser.username

    # VCS 設定を取得
    $VcsSetting = Get-TeamCityUserProperty $TeamCityUrl $User $Password "id:${UserId}" "plugin:vcs:anyVcs:anyVcsRoot"

    # メールアドレスを取得
    $Email = Get-TeamCityUserField $TeamCityUrl $User $Password "id:${UserId}" "email"

    # メールアドレスが設定されているかどうかを確認
    if ($Email -eq "")
    {
        Write-Warning "Warning: ${UserName} does NOT have email setting."
    }
    else
    {
        # メールアカウントを取得
        $EmailAccount = $Email.Split('@')[0]

        # VCS 設定がメールアカウントでなければ、メールアカウントで上書き
        if ($EmailAccount -ne $VcsSetting)
        {
            # 設定を書き換え
            $ChangedVcsSetting = Set-TeamCityUserProperty $TeamCityUrl $User $Password "id:${UserId}" "plugin:vcs:anyVcs:anyVcsRoot" $EmailAccount
            if ($ChangedVcsSetting -eq $EmailAccount)
            {
                Write-Output "Info: ${UserName}'s VCS Setting was changed. (${VcsSetting} -> ${ChangedVcsSetting})"
            }
            else
            {
                Write-ErrorMessage "Error: ${UserName}'s VCS Setting was NOT changed successfully."
            }
        }
    }
}
