﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Compare files in two directories.

    .DESCRIPTION
        Compare files in two directories.
#>
param
(
    [parameter(Mandatory=$True)][string] $Dir1,
    [parameter(Mandatory=$True)][string] $Dir2,
    [string] $OutputDir,
    [switch] $TeamCity,
    [string] $TestName = "CompareProcess"
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${ScriptDirectoryPath}\Modules\Error"
Import-Module "${ScriptDirectoryPath}\Modules\Path"

function Make-SymbolTable
{
    param
    (
        [string] $File
    )

    $symbolTable = @()
    $lineCount = 0
    $isSymbolFound = $false

    foreach ($l in (Get-Content $File))
    {
        if ($l -match "^([0-9a-f]+) <(.+)>:")
        {
            # シンボルのシグネチャが見つかった
            $isSymbolFound = $true

            # Matches[1] = address, $Matches[2] = name
            $symbolName = $Matches[2];

            # 行数カウントをリセット
            $lineCount = 0
        }
        elseif ($l -match "^\s*$")
        {
            # 空行。シンボルが見つかっていたらその情報を書き込んでおく
            if ($isSymbolFound)
            {
                # シンボルテーブルとしてシンボル名と行数の情報を登録
                $symbolTable += @{
                    name = ${symbolName};
                    count = ${lineCount};
                }

                $isSymbolFound = $false
            }
        }
        else
        {
            # シンボルが見つかった以降の空行以外の行はカウントしておく
            if ($isSymbolFound)
            {
                $lineCount += 1
            }
        }
    }

    $symbolTable
}

function Compare-SymbolTables
{
    param
    (
        $table1,
        $table2
    )

    $output = ""
    $isFirstDifference = $true

    foreach ($base in $table1)
    {
        $isDifferenceFound = $false

        foreach ($target in $table2)
        {
            if ($base.name -eq $target.name)
            {
                if ($base.count -eq $target.count)
                {
                    # 同一のものが見つかったので検索は打ち止め break
                    $isDifferenceFound = $false
                    break
                }
                else
                {
                    # 差分は見つかった。が、オーバーロードにより同じシンボル名のものがあるかもしれないので引き続き検索は続ける
                    $isDifferenceFound = $true
                    $diff = @{
                        name = $base.name
                        baseCount = $base.count
                        targetCount = $target.count
                    }
                }
            }
        }

        if ($isDifferenceFound)
        {
            if ($isFirstDifference)
            {
                # 最初の差分に対してはヘッダ情報を出力
                $output += "`n{0, -80} {1, 10}" -f "Symbol name","Line number difference`n"
                $output += "{0, -80} {1, 10}" -f "-----------","----------------------`n"

                $isFirstDifference = $false
            }

            # 名前や行数の差分を整形して出力
            $output += "{0, -80} {1, 4}  => {2, 4}`n" -f $diff.name,$diff.baseCount,$diff.targetCount
        }
    }

    $output
}

if (!(Test-Path $Dir1 -PathType Container) -Or !(Test-Path $Dir2 -PathType Container))
{
    Write-Host "Error: input directory does not found."
    exit 1
}

# 出力先ディレクトリが与えられたら作成する
if ($OutputDir)
{
    if (Test-Path $OutputDir)
    {
        Remove-Item -Path $OutputDir -Recurse -Force
    }
    New-Item -Path $OutputDir -ItemType Directory -Force | Out-Null
}

foreach ($file1 in (Get-ChildItem $Dir1))
{
    # .dasm 以外はスキップ
    if ($file1.Extension -ne ".dasm")
    {
        continue
    }

    $fileName1 = $file1.Name
    $procName1 = $file1.BaseName -replace "\.[0-9a-f]{7,}.*$","" # リビジョンっぽい文字列以降は無視する
    $isProcFound = $false

    # start marker for CI
    if ($TeamCity)
    {
        Write-Host "##teamcity[testStarted name='$TestName.$procName1']"
    }

    foreach ($file2 in (Get-ChildItem $Dir2))
    {
        # 同じく .dasm 以外はスキップ
        if ($file2.Extension -ne ".dasm")
        {
            continue
        }

        $fileName2 = $file2.Name
        $procName2 = $file2.BaseName -replace "\.[0-9a-f]{7,}.*$",""

        # ファイル名からリビジョンっぽい文字列以降を取り除いたプロセス名が一致したら比較する
        if ($procName1 -eq $procName2)
        {
            $isProcFound = $true
            $isDumpFileFound = $false
            $differenceBody = ""

            # full-contents ファイルがあればまずそれを比較し、差分があったら逆アセンブルファイルを比較する
            $dumpFile1 = $fileName1 -replace "\.dasm$",".dump"
            $dumpFile2 = $fileName2 -replace "\.dasm$",".dump"
            if ((Test-Path "$Dir1/$dumpFile1") -and (Test-Path "$Dir2/$dumpFile2"))
            {
                $isDumpFileFound = $true
                $diff = (& $(Get-GitPath) diff --no-index --word-diff $Dir1/$dumpFile1 $Dir2/$dumpFile2)
                if (! $diff)
                {
                    if ($TeamCity)
                    {
                        Write-Host "No change is found"
                    }
                    break
                }
            }

            $diff = (& $(Get-GitPath) diff --no-index --word-diff $Dir1/$fileName1 $Dir2/$fileName2)
            if ($diff)
            {
                $messageBody = "Files $Dir1/$fileName1 and $Dir2/$fileName2 differ"

                $separator = "|n-------------------------------------------------------------------------------|n"
                if ($diff.Length -lt 500)
                {
                    # 差分レベル1：
                    #
                    # diff の行数が小さい＝全体の差分が小さい
                    $differenceLevel = "Low"
                    $shortDescription = "Number of different lines is small ({0} lines)" -f $diff.Length
                    $possibleReason = "1. Fix is really simple|n2. Changes are only in static values|n3. Only debug sections are different"

                    # ログには diff の結果を出力する
                    $differenceBody = $diff
                }
                else
                {
                    # 差分の行数が大きかったらシンボル情報を抽出して行数が変わったシンボルがあるか比較してみる
                    $content1 = (Make-SymbolTable $Dir1/$fileName1)
                    $content2 = (Make-SymbolTable $Dir2/$fileName2)

                    $differenceBody = Compare-SymbolTables $content1 $content2
                    if ($differenceBody)
                    {
                        # 差分レベル2：
                        #
                        # diff の行数が大きいがシンボルのサイズが変わったものが検出されている
                        # それによるセクションサイズのずれ＝オフセットのズレと思われる
                        $differenceLevel = "Middle"
                        $shortDescription = "Number of different lines is large ({0} lines) but symbol size difference are found" -f $diff.Length
                        $possibleReason = "Section size/offset has been changed by such symbols"
                    }
                    else
                    {
                        # 差分レベル3：diff の行数が大きいが、しかしサイズが変わったシンボルは検出されていない
                        #
                        # ビルドオプションの違いなど関数のリオーダーが発生するような何かが起こっている
                        # 実際の差分を調べて問題がないことを確認しないといけない
                        $differenceLevel = "High"
                        $shortDescription = "Number of different lines is large ({0} lines) and no symbol size difference is found" -f $diff.Length
                        $possibleReason = "Re-ordering happens somehow. Please check actual dasm files"
                    }
                }

                $detailsBody = "{0}Difference level : {1}|n|nShort description:|n{2}|n|nPossible reasons:|n{3}{4}" -f $separator,$differenceLevel,$shortDescription,$possibleReason,$separator

                if ($TeamCity)
                {
                    Write-Host "##teamcity[testFailed name='$TestName.$procName1' message='$messageBody' details='$detailsBody']"
                }
                else
                {
                    # 改行コードを変換しつつ message と details をログ表示
                    $messageBody -replace "\|n","`n"
                    $detailsBody -replace "\|n","`n"
                }
                $differenceBody

                # 出力先ディレクトリが与えられたら差分があった逆アセンブルファイルをコピー
                if ($OutputDir)
                {
                    Copy-Item $Dir1/$fileName1 $OutputDir/$fileName1
                    if ($fileName1 -eq $fileName2)
                    {
                        # ファイル名が同じだったら名前を変えてコピーしてあげる
                        $target = "{0}(1){1}" -f $file2.BaseName,$file2.Extension
                        Copy-Item $Dir2/$fileName2 $OutputDir/$target
                    }
                    else
                    {
                        Copy-Item $Dir2/$fileName2 $OutputDir/$fileName2
                    }

                    # ダンプファイルがあったらそれもコピー
                    if ($isDumpFileFound)
                    {
                        Copy-Item $Dir1/$dumpFile1 $OutputDir/$dumpFile1
                        if ($dumpFile1 -eq $dumpFile2)
                        {
                            # ファイル名が同じだったら名前を変えてコピーしてあげる
                            $target = "{0}(1){1}" -f $file2.BaseName,".dump"
                            Copy-Item $Dir2/$dumpFile2 $OutputDir/$target
                        }
                        else
                        {
                            Copy-Item $Dir2/$dumpFile2 $OutputDir/$dumpFile2
                        }
                    }
                }
            }
            else
            {
                if ($TeamCity)
                {
                    Write-Host "No change is found"
                }
            }

            break
        }
    }

    if (!$isProcFound)
    {
        if ($TeamCity)
        {
            Write-Host "##teamcity[testFailed name='$TestName.$procName1' message='No dasm is found correspond to $fileName1']"
        }
    }

    # end marker for CI
    if ($TeamCity)
    {
        Write-Host "##teamcity[testFinished name='$TestName.$procName1']"
    }
}
