﻿#r "System.Web.Extensions"

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Web.Script.Serialization;

private const string ProjectAssetsJsonName = "project.assets.json";
private const string ProjectAssetsJsonPattern = "project.assets.json";
private const string NupkgPattern = "*.nupkg";
private const string NupkgExtension = ".nupkg";

public static IEnumerable<string> EnumeratePackages(string searchDirectoryOrFile)
{
    if (Directory.Exists(searchDirectoryOrFile))
    {
        return EnumeratePackagesInDirectory(searchDirectoryOrFile);
    }
    else if (File.Exists(searchDirectoryOrFile))
    {
        return EnumeratePackagesFromFile(searchDirectoryOrFile);
    }
    else
    {
        throw new MyException($"{searchDirectoryOrFile} not found.");
    }
}

public static IEnumerable<string> EnumeratePackagesInDirectory(string searchDirectory)
{
    foreach (var nupkgPath in Directory.EnumerateFiles(searchDirectory, NupkgPattern, SearchOption.AllDirectories))
    {
        yield return nupkgPath;
    }

    foreach (var lockFilePath in Directory.EnumerateFiles(searchDirectory, ProjectAssetsJsonPattern, SearchOption.AllDirectories))
    {
        foreach (var nupkgPath in EnumeratePackagesInLockFile(lockFilePath))
        {
            yield return nupkgPath;
        }
    }
}

public static IEnumerable<string> EnumeratePackagesFromFile(string filePath)
{
    if (EqualIgnoreCase(Path.GetFileName(filePath), ProjectAssetsJsonName))
    {
        return EnumeratePackagesInLockFile(filePath);
    }
    else if (EqualIgnoreCase(Path.GetExtension(filePath), NupkgExtension))
    {
        return new string[] { filePath };
    }
    else
    {
        throw new MyException($"{filePath} : unknown kind of file.");
    }
}

public static IEnumerable<string> EnumeratePackagesInLockFile(string lockFilePath)
{
    try
    {
        return EnumeratePackagesInLockFileImpl(lockFilePath);
    }
    catch (MyException e)
    {
        throw new MyException($"{lockFilePath} : {e.Message}", e);
    }
}

public static IEnumerable<string> EnumeratePackagesInLockFileImpl(string lockFilePath)
{
    var assets = new ProjectAssets(lockFilePath);

    foreach (var library in assets.Libraries)
    {
        if (library.Type == "package")
        {
            var packageDirectory = ResolvePackageDirectory(assets.PackageFolders, library.Path);
            var packages = Directory.GetFiles(packageDirectory, NupkgPattern, SearchOption.TopDirectoryOnly);
            switch (packages.Length)
            {
                case 1:
                    yield return packages[0];
                    break;
                case 0:
                    throw new MyException($"{packageDirectory} contains no nupkg files.");
                default:
                    throw new MyException($"{packageDirectory} contains multiple nupkg files.");
            }
        }
    }
}

public static string ResolvePackageDirectory(IEnumerable<string> packageFolders, string libraryPath)
{
    foreach (var packageFolder in packageFolders)
    {
        var candidate = Path.Combine(packageFolder, libraryPath);
        if (Directory.Exists(candidate))
        {
            return Path.GetFullPath(candidate);
        }
    }

    throw new MyException(string.Format("Package '{0}' not found.", libraryPath));
}

/// <summary>
/// project.assets.json のデータ構造の一部を表現する。
/// </summary>
internal class ProjectAssets
{
    private static readonly int[] ExpectedVersions = new int[] { 2, 3 };

    public IReadOnlyCollection<string> PackageFolders { get; }
    public IReadOnlyCollection<Library> Libraries { get; }

    public ProjectAssets(string lockFilePath)
    {
        var text = File.ReadAllText(lockFilePath);
        var serializer = new JavaScriptSerializer();
        IReadOnlyDictionary<string, object> assets;
        try
        {
            assets = serializer.Deserialize<IReadOnlyDictionary<string, object>>(text);
        }
        catch (Exception e) when (e is ArgumentException || e is InvalidOperationException)
        {
            throw new MyException($"Failed to deserialize JSON : {e.Message}", e);
        }

        var version = GetValue<int>(assets, "version", "(root)");
        if (!ExpectedVersions.Contains(version))
        {
            var expectedVersionsString = string.Join(", ", ExpectedVersions.Select(x => x.ToString()));
            throw new MyException($"unknown version {version}. (expected: {expectedVersionsString})");
        }

        this.PackageFolders =
            GetValue<IReadOnlyDictionary<string, object>>(assets, "packageFolders", "(root)")
            .Keys
            .ToArray();

        this.Libraries =
            GetValue<IReadOnlyDictionary<string, object>>(assets, "libraries", "(root)")
            .Select(kv => new Library((IReadOnlyDictionary<string, object>)kv.Value, kv.Key))
            .ToArray();
    }

    public class Library
    {
        public string Type { get; }
        public string Path { get; }

        public Library(IReadOnlyDictionary<string, object> libraryElement, string name)
        {
            this.Type = GetValue<string>(libraryElement, "type", name);
            this.Path = GetValue<string>(libraryElement, "path", name);
        }
    }
}

internal static T GetValue<T>(IReadOnlyDictionary<string, object> dic, string key, string parentKey)
{
    object value;
    if (dic.TryGetValue(key, out value))
    {
        if (!(value is T))
        {
            throw new MyException($"Value '{value}' of key '{key}' under '{parentKey}' is not of type '{typeof(T).Name}'.");
        }

        return (T)value;
    }
    else
    {
        throw new MyException($"Key '{key}' not found under '{parentKey}'.");
    }
}

internal static bool EqualIgnoreCase(string x, string y)
{
    return StringComparer.OrdinalIgnoreCase.Compare(x, y) == 0;
}

internal static string NormalizePath(string path)
{
    try
    {
        return Path.GetFullPath(path);
    }
    catch (Exception e) when (e is ArgumentException || e is NotSupportedException)
    {
        throw new MyException($"{path} : {e.Message}", e);
    }
}

[Serializable]
public class MyException : Exception
{
    public MyException() { }
    public MyException(string message) : base(message) { }
    public MyException(string message, Exception inner) : base(message, inner) { }
    protected MyException(
        System.Runtime.Serialization.SerializationInfo info,
        System.Runtime.Serialization.StreamingContext context) : base(info, context) { }
}

//
// EnumeratePackages <search path> ...
//    指定のパス配下の .nupkg のパスおよび
//    指定のパス配下の project.assets.json で参照されているパッケージの .nupkg のパスを標準出力に出力します
//
var packagePaths = new List<string>();

foreach (var packageSearchPath in Args)
{
    try
    {
        packagePaths.AddRange(EnumeratePackages(NormalizePath(packageSearchPath)));
    }
    catch (Exception e) when (e is MyException || e is IOException)
    {
        Console.Error.WriteLine(e.Message);
        Environment.Exit(1);
    }
}

packagePaths.Sort(StringComparer.OrdinalIgnoreCase);
foreach (var s in packagePaths.Distinct(StringComparer.OrdinalIgnoreCase))
{
    Console.WriteLine(s);
}

Environment.Exit(0);
