﻿Param([string]$vsixDir, [string]$install)

$scriptPath = $MYInvocation.MyCommand.Path
$scriptDir = Split-Path -Parent $scriptPath

function GetVsixInstallerPathWithVswhere($majorVer) {
    $vswherePath = "C:\Program Files (x86)\Microsoft Visual Studio\Installer\vswhere.exe"
    if (Test-Path $vswherePath) {
        # do nothing
    } else {
        return 88802
    }
    $nextMajorVer = [int]$majorVer + 1
    $versionRangeStr = "[$majorVer.0,$nextMajorVer.0`)"
    $vsInstallDir = & "$vswherePath" -format value -products Microsoft.VisualStudio.Product.Professional -version $versionRangeStr -property installationPath
    if (($vsInstallDir -ne $null) -And (Test-Path $vsInstallDir)) {
        # do nothing
    } else {
        $vsInstallDir = & "$vswherePath" -format value -products Microsoft.VisualStudio.Product.Community -version $versionRangeStr -property installationPath
        if (($vsInstallDir -ne $null) -And (Test-Path $vsInstallDir)) {
            # do nothing
        } else {
            return 88801
        }
    }
    $ret = "$vsInstallDir\Common7\IDE\VSIXInstaller.exe"
    return $ret
}

function GetVsixInstallerPathWithRegistry($version) {
    $reg_result = reg query "HKLM\SOFTWARE\Microsoft\VisualStudio\$version" /reg:32 /v "InstallDir" 2>$null
    if ($LASTEXITCODE -ne 0) {
        return 88801
    }
    $vsInstallDir = $reg_result[2] -replace " *InstallDir *REG_SZ *",""
    if (Get-Item -Path $vsInstallDir) {
        # do nothing
    } else {
        return 88801
    }
    $ret = "${vsInstallDir}VSIXInstaller.exe"
    return $ret
}

function GetVsixInstallerPath([string]$versionStr) {
    $majorVer = $versionStr.substring(0,2)
    $ret = "unknown"
    if ($majorVer -ge 15) {
        $ret = GetVsixInstallerPathWithVswhere($majorVer)
    } elseif ($majorVer -le 11) {
        $ret = 88803
    } else {
        $ret = GetVsixInstallerPathWithRegistry($versionStr)
    }
    return $ret
}

function GetVsiMsiPath([string]$versionStr) {
    if ($vsixDir -eq "") {
        $vsixDir = "$scriptDir\.."
    }
    $major_ver = [int]$versionStr
    $vsixFile = Get-ChildItem $vsixDir -Filter VsiForNX_v$major_ver*.msi
    $vsixFilePath = "$vsixDir\$vsixFile"

    return $vsixFilePath
}

function ConvertVsixExitCodeToMessage($exitCode) {
    $acceptCode = @{
        0     = "[OK] Success";
        1001  = "[OK] ExtensionManager.AlreadyInstalledException";
        1002  = "[OK] ExtensionManager.NotInstalledException";
        2003  = "[OK] VSIXInstaller.NoApplicableSKUsException";
        88801 = "[OK] Cannot find VisualStudio install path";
        88802 = "[OK] Cannot find vswhere (VS2017 not installed maybe)";
        88803 = "[OK] Not supported version";

        2001  = "[NG] VSIXInstaller.InvalidCommandLineException";
        2002  = "[NG] VSIXInstaller.InvalidLicenseException";
        2004  = "[NG] VSIXInstaller.BlockingProcessesException";
    }

    if ($acceptCode.Contains($exitCode)) {
        return $acceptCode[$exitCode]
    } else {
        return "[NG] Error"
    }
}

function ConvertMsiExitCodeToMessage($exitCode) {
    $acceptCode = @{
        0     = "[OK] Success";

        1618  = "[NG] Install already running";
    }

    if ($acceptCode.Contains($exitCode)) {
        return $acceptCode[$exitCode]
    } else {
        return "[NG] Error"
    }
}

# Uninstall Oasis 5.0 and later
function UninstallVsix{
    Param([string]$vsixInstaller, [string]$version)

    $oasis_guid = @{
        12 = "DE7A8DA1-B76C-4429-825F-69E72DA7915A";
        14 = "DE7A3ACC-BD6B-41B5-9ACA-74B7D0AB7B34";
        15 = "DE7A3728-43F6-4A2E-BD0E-5252AF2844A3";
    }

    $major_version = [int]$version
    return UninstallVsixImpl $vsixInstaller $version $oasis_guid[$major_version]
}

# Uninstall Oasis 4.x and before
function UninstallOldVsix{
    Param([string]$vsixInstaller, [string]$version)
    return UninstallVsixImpl $vsixInstaller $version "FBEDE382-A024-4D45-8D3D-029D3573F7CC"
}

function UninstallVsixImpl{
    Param([string]$vsixInstaller, [string]$version, [string]$guid)

    $skuName = "Pro"
    if ($vsixInstaller -match "Community") {
        $skuName = "Community"
    }

    ### アンインストール
    $retry = 0
    while ($retry -lt 3)
    {
        $option = "/q /skuName:$skuName /skuVersion:$version /u:$guid"
        Write-Output "[UNINSTALL] `"$vsixInstaller`" $option"
        $proc = Start-Process -FilePath "$vsixInstaller" -ArgumentList "$option" -Wait -PassThru
        $exitCode = $proc.ExitCode
        $msg = ConvertVsixExitCodeToMessage($exitCode)
        Write-Output "=> $exitCode ($msg)`r`n"

        # Visual Studio が終了しきっていないケースで 2004 が返る様子。なので 3 回までリトライする。
        if ($exitCode -eq 2004) {
            $retry++
        } else {
            break
        }
    }

    return $exitCode
}

function FindUninstallItemWithHkey ([string]$displayNameToSearch, [string]$hkey) {
    $keyToCheck = "${hkey}:\software\microsoft\windows\currentversion\uninstall"
    if (-not (Test-Path $keyToCheck)) {
        return ""
    }

    $uninstallItems = Get-ChildItem $keyToCheck
    foreach ($item in $uninstallItems)
    {
        $displayName = [string]$item.GetValue("DisplayName")
        if ($displayName.Contains($displayNameToSearch))
        {
            return $item.pspath
        }
    }
    return ""
}

function FindUninstallItem ([string]$displayNameToSearch) {
    $ret = FindUninstallItemWithHkey $displayNameToSearch "hklm"
    if ($ret -eq "")
    {
        $ret = FindUninstallItemWithHkey $displayNameToSearch "hkcu"
    }
    return $ret
}

function CleanUpVsiMsi([string]$installedItem)
{
    Remove-Item $installedItem

    $ProgramFilesX86 = [Environment]::ExpandEnvironmentVariables( "%ProgramFiles(x86)%" )
    $isTmInstalled = FindUninstallItem("Nintendo Target Manager")
    if ($isTmInstalled -eq "")
    {
        Write-Output "Removing $ProgramFilesX86\MSBuild\Microsoft.Cpp\v4.0\v140\Platforms\NX*"
        $PathsToDelete = "$ProgramFilesX86\MSBuild\Microsoft.Cpp\v4.0\v140\Platforms\NX32", "$ProgramFilesX86\MSBuild\Microsoft.Cpp\v4.0\v140\Platforms\NX64"
        foreach ($path in $PathsToDelete) {
            if (Test-Path $path) {
                Remove-Item -Recurse -Force $path
            }
        }
    }
    else
    {
        Write-Output "Skip removing $ProgramFilesX86\MSBuild\Microsoft.Cpp\v4.0\v140\Platforms\NX*"
    }
}

function UninstallMsiImpl ([string]$version) {
    $majorVer = [int]$version
    switch ($majorVer)
    {
        12 { $displayNameToSearch = "Nintendo NX Visual Studio 2013 Integration"}
        14 { $displayNameToSearch = "Nintendo NX Visual Studio 2015 Integration"}
        default {return}
    }

    $installedItem = FindUninstallItem $displayNameToSearch
    if ($installedItem -eq "")
    {
        Write-Output "[UNINSTALL] VSI $version via msi"
        Write-Output "=> 0 ([OK] $displayNameToSearch is not found.)`r`n"
        return 0
    }

    $produceCode = Split-Path -Path $InstalledItem -Leaf
    $exe = "msiexec.exe"
    $time = Get-Date -UFormat "%Y-%m-%d-%H%M%S"
    $logFileName = $env:TEMP + "/VsiUninstall_" + $time + ".log"
    $option = "/x $produceCode /l*v `"$logFileName`" /passive /promptrestart"
    Write-Output "[UNINSTALL] $exe $option"
    $proc = Start-Process -FilePath $exe -ArgumentList $option -Wait -PassThru
    $exitCode = $proc.ExitCode
    if ($exitCode -eq 1603)
    {
        Write-Output "Found corrupted installation. Cleaning up..."
        CleanUpVsiMsi $installedItem
        $exitCode = 0
    }
    $msg = ConvertVsixExitCodeToMessage($exitCode)
    Write-Output "=> $exitCode ($msg)`r`n"
    return $exitCode
}

function InstallVsixImpl {
    Param([string]$vsixInstaller, [string]$version)

    $version_float = [single]$version
    $skuName = "Pro"
    if ($vsixInstaller -match "Community") {
        $skuName = "Community"
    }

    ### インストール
    $retry = 0
    while ($retry -lt 3)
    {
        if ($vsixDir -eq "") {
            $vsixDir = "$scriptDir\.."
        }
        $major_ver = [int]$version
        $vsixFile = Get-ChildItem $vsixDir -Filter VsiForNX_v$major_ver*.vsix
        $vsixFilePath = "$vsixDir\$vsixFile"
        $option = ""
        if ($version_float -ge 15.0) {
            $devenv_path = (Split-Path $vsixInstaller -Parent) + "\devenv.exe"
            $option ="/appidinstallpath:`"$devenv_path`" /appidname:`"Microsoft Visual Studio`""
        }
        $option = "/admin /q /skuName:$skuName /skuVersion:$version `"$vsixFilePath`""
        Write-Output "[INSTALL] `"$vsixInstaller`" $option"
        $proc = Start-Process -FilePath "$vsixInstaller" -ArgumentList "$option" -Wait -PassThru
        $exitCode = $proc.ExitCode
        $msg = ConvertVsixExitCodeToMessage($exitCode)
        Write-Output "=> $exitCode ($msg)`r`n"

        # Visual Studio が終了しきっていないケースで 2004 が返る様子。なので 3 回までリトライする。
        if ($exitCode -eq 2004) {
            $retry++
        } elseif ($msg -match "^\[NG\]") {
            return 1
        } else {
            break
        }
    }
    return 0
}

function InstallVsiByMsi ([string]$msiPath) {
    $exe = "msiexec.exe"
    $time = Get-Date -UFormat "%Y-%m-%d-%H%M%S"
    $logFileName = $env:TEMP + "/VsiInstall_" + $time + ".log"
    $option = "/i `"$msiPath`" /l*v `"$logFileName`" /passive /promptrestart"
    Write-Output "$exe $option"
    $proc = Start-Process -FilePath $exe -ArgumentList $option -Wait -PassThru
    $exitCode = $proc.ExitCode
    Write-Output "=> $exitCode`r`n"
    return $exitCode
}

function GetInstalledVsi2015MsiVersion([string]$hkey) {
    $keyToCheck = "${hkey}:\software\microsoft\windows\currentversion\uninstall"
    if (-not (Test-Path $keyToCheck)) {
        return $null
    }

    $uninstallItems = Get-ChildItem $keyToCheck
    foreach ($item in $uninstallItems)
    {
        $displayName = [string]$item.GetValue("DisplayName")
        if ($displayName.Contains("Nintendo NX Visual Studio 2015 Integration"))
        {
            return $item.GetValue("DisplayVersion")
        }
    }
    return $null
}

function NeedsToCheckDrive () {
    $version = GetInstalledVsi2015MsiVersion "hklm"
    if ($version -eq $null)
    {
        $version = GetInstalledVsi2015MsiVersion "hkcu"
    }

    if ($version -eq $null) {
        return $FALSE
    }

    return ([System.Version]"$version" -lt [System.Version]"5.2.2.22884")
}

# Return drive letter of the volume which has drive letter and the largest remaining size
function GetFreestDrive()
{
    $maxSize = 0
    $driveLetter = $null
    foreach ($volume in Get-WmiObject -Class Win32_LogicalDisk -Filter "DriveType=3" -ComputerName .) {
        if ($volume.FreeSpace -gt $maxSize) {
            $maxSize = $volume.FreeSpace
            $driveLetter = $volume.DeviceID
        }
    }
    return $driveLetter.Substring(0, 1)
}

function DriveHasFile([string]$driveToCheck) {
    return (@(Get-ChildItem "${driveToCheck}:\" -Force | Where-Object{!$_.PSIsContainer -and !($_.Attributes -band [System.IO.FileAttributes]::System)}).length -ne 0)
}

# ==============================================================================
# メイン処理
# ==============================================================================
## VS の起動チェック
& "$scriptDir/check_vs_running.bat"
$ret = $LASTEXITCODE
if ($ret -ne 0) {
    exit $ret
}

## Check drive if necessary
if (NeedsToCheckDrive) {
    $driveToCheck = GetFreestDrive
    while (DriveHasFile $driveToCheck) {
        & "$scriptDir/check_drive.bat" $driveToCheck
        $ret = $LASTEXITCODE
        if ($ret -eq 2) { # Cancel
            exit $ret
        }
        if ($ret -eq 5) { # Ignore (removed from check_drive.vbs so must not come)
            break
        }
    }
}

Write-Output "-------------------------------------------------------------"
Write-Output "Uninstall"
Write-Output "-------------------------------------------------------------"
## VSIX のアンインストール
$result_hash = @{}
@("12.0", "14.0") | ForEach-Object -Process {
    $Version = $_

    Write-Output "Version: $Version"

    $VsixInstallerPath = GetVsixInstallerPath($Version)
    if (Test-Path $VsixInstallerPath) {
        $result = UninstallOldVsix -vsixInstaller $VsixInstallerPath -version $Version
        $result_hash.Add("UninstallViaOldVsix-$Version", $result[-1])
        $result[-1] = $null
        Write-Output $result

        $result = UninstallVsix -vsixInstaller $VsixInstallerPath -version $Version
        $result_hash.Add("UninstallViaVsix-$Version", $result[-1])
        $result[-1] = $null
        Write-Output $result
    }
    $result = UninstallMsiImpl -version $Version
    $result_hash.Add("UninstallViaMsi-$Version", $result[-1])
    $result[-1] = $null
    Write-Output $result
}

@("15.0") | ForEach-Object -Process {
    $Version = $_

    Write-Output "Version: $Version"
    $VsixInstallerPath = GetVsixInstallerPath($Version)
    Write-Output "[VSIX Installer Path] ($Version) ($VsixInstallerPath)`r`n"
    if (Test-Path $VsixInstallerPath) {
        $result = UninstallOldVsix -vsixInstaller $VsixInstallerPath -version $Version
        $result_hash.Add("UninstallViaOldVsix-$Version", $result[-1])
        $result[-1] = $null
        Write-Output $result

        $result = UninstallVsix -vsixInstaller $VsixInstallerPath -version $Version
        $result_hash.Add("UninstallViaVsix-$Version", $result[-1])
        $result[-1] = $null
        Write-Output $result
    }
}

## Check the result of uninstallation and exit if failure
$result = 0
foreach ($k in $result_hash.Keys) {
    $ret = $result_hash[$k]
    Write-Output "[$k] => $ret"
    if ($k.IndexOf("Vsix") -ne -1) {
        $msg = ConvertVsixExitCodeToMessage($ret)
    }
    else {
        $msg = ConvertMsiExitCodeToMessage($ret)
    }

    if ($msg.IndexOf("[NG]") -eq 0) {
        $result = $ret
    }
}
if ($result -ne 0)
{
    exit $result
}

if ($install -ne "")
{
    ## VSIX のインストール
    Write-Output "-------------------------------------------------------------"
    Write-Output "Install"
    Write-Output "-------------------------------------------------------------"

    $result_hash = @{}
    @("14.0") | ForEach-Object -Process {
        $Version = $_

        Write-Output "Version: $Version"

        $VsiMsiPath = GetVsiMsiPath($Version)
        $result = InstallVsiByMsi $VsiMsiPath
        $result_hash.Add("Install-$Version", $result[-1])
        $result[-1] = $null
        Write-Output $result
    }

    @("15.0") | ForEach-Object -Process {
        $Version = $_

        Write-Output "Version: $Version"

        $VsixInstallerPath = GetVsixInstallerPath($Version)
        Write-Output "[VSIX Installer Path] ($Version) ($VsixInstallerPath)`r`n"
        if (Test-Path $VsixInstallerPath) {
            $result = InstallVsixImpl -vsixInstaller $VsixInstallerPath -version $Version
            $result_hash.Add("Install-$Version", $result[-1])
            $result[-1] = $null
            Write-Output $result
        }
    }

    $result = 0
    foreach ($k in $result_hash.Keys) {
        $ret = $result_hash[$k]
        Write-Output "[$k] => $ret"
        if ($ret -ne 0) {
            $result = $ret
        }
    }
}

exit $result
